// SPDX-License-Identifier: MIT

use anyhow::Result;
use ipnet::Ipv4Net;
use std::collections::HashMap;
use std::fs;
use std::path::Path;

mod config;
pub mod container;
mod dhcp;
mod html_template;
mod iptables;
mod logged_in;
pub mod networkmanager;
mod password;
mod process;
mod slice_ext;
mod title;
pub mod usb_filter;
mod variant;
mod ws;

#[cfg(feature = "restapi")]
mod auth_header;
#[cfg(feature = "restapi")]
pub mod restapi;

pub use config::{Config, Customize, SideMenu};
pub use dhcp::{dhcp_info, dhcp_list, DhcpInfo};
pub use html_template::{filters, HtmlTemplate, RawTemplate};
pub use iptables::{IptablesInfo, NatInfo, PortForwardingInfo};
pub use logged_in::{check_auth, LoggedIn};
pub use password::{check_password, current_password_hash, password_throttle_init};
pub use process::{
    exec_command, exec_command_no_script, exec_command_stdin, exec_command_with_opts,
    stream_command, stream_command_input, CommandOpts,
};
pub use slice_ext::SliceExt;
pub use title::{get_title, Title};
pub use variant::Variant;

#[cfg(feature = "restapi")]
#[allow(unused_imports)]
pub use restapi::{
    json_response, json_response_raw, json_stream_command, multipart_to_input_chan, permissions::*,
    CheckAuthRestApi, JsonOrForm, JsonOrFormOption, RestApiSerializeToken, RestApiToken,
    SerializeTokenOptions,
};

pub const WWAN_CON_NAME: &str = "gsm-ttyCommModem";
pub const AP_BRIDGE_NAME: &str = "abos_web_br_ap";
pub const VERSION: &str = env!("CARGO_PKG_VERSION");
pub const OPENVPN_CONTAINER: &str = "abos_web_openvpn";

pub fn ap_interface() -> Option<&'static str> {
    if Path::new("/sys/class/net/uap0").is_dir() {
        Some("uap0")
    } else if Path::new("/sys/class/net/wlan0").is_dir() {
        Some("wlan0")
    } else {
        None
    }
}

pub fn wlan_interface() -> Option<&'static str> {
    if Path::new("/sys/class/net/mlan0").is_dir() {
        Some("mlan0")
    } else if Path::new("/sys/class/net/wlan0").is_dir() {
        Some("wlan0")
    } else {
        None
    }
}

pub fn wwan_interface() -> Option<&'static str> {
    if Path::new("/sys/class/net/usb0").is_dir() {
        Some("usb0")
    } else if Path::new("/sys/class/net/ppp0").is_dir() {
        Some("ppp0")
    } else {
        None
    }
}

pub fn wwan_module() -> Option<&'static str> {
    if Path::new("/sys/devices/platform/ec25-reset").is_dir() {
        Some("ec25")
    } else if Path::new("/sys/devices/platform/els31-reset").is_dir() {
        Some("els31")
    } else if Path::new("/sys/devices/platform/ems31-reset").is_dir() {
        Some("ems31")
    } else if Path::new("/sys/devices/platform/sim7672-reset").is_dir() {
        Some("sim7672")
    } else if Path::new("/sys/devices/platform/em05g-reset").is_dir() {
        Some("em05g")
    } else {
        None
    }
}

pub fn network_interfaces() -> Vec<String> {
    let mut v = Vec::new();
    if let Ok(entries) = fs::read_dir("/sys/class/net") {
        for entry in entries.flatten() {
            v.push(entry.file_name().to_str().unwrap_or_default().to_string());
        }
    }
    v
}

pub fn exists_eth1() -> bool {
    Path::new("/sys/class/net/eth1").is_dir()
}

pub async fn current_address<'a>(
    dev_name: &str,
    interface: &str,
) -> Result<HashMap<&'a str, String>, &'static str> {
    let args = &["get_address.sh", dev_name, interface];
    let output = match exec_command(args).await {
        Ok(o) => o,
        Err(_) => return Err("Could not get current address."),
    };
    let stdout = String::from_utf8_lossy(&output.stdout);
    // addrs[0]: ip addr with netmask e.g. 192.168.1.1/24
    // addrs[1]: gateway addr
    // addrs[2]: DNS addr
    // addrs[3]: MAC addr
    let addrs = stdout
        .split(';')
        .collect::<Vec<&str>>()
        .iter()
        .map(|s| s.trim().to_string())
        .collect::<Vec<String>>();

    let na = "--".to_string();
    let mut map = HashMap::new();

    map.insert("current_ip_addr", na.clone());
    map.insert("current_netmask", na.clone());
    map.insert("current_gw_addr", na.clone());
    map.insert("current_dns_addr", na.clone());
    map.insert("mac_addr", na.clone());
    map.insert("interface", na);

    if !addrs[0].is_empty() {
        let net: Ipv4Net = match addrs[0].parse() {
            Ok(n) => n,
            Err(_) => return Err("Invalid IP Address."),
        };
        map.insert("current_ip_addr", net.addr().to_string());
        map.insert("current_netmask", net.netmask().to_string());
    }
    if !addrs[1].is_empty() {
        map.insert("current_gw_addr", addrs[1].to_string());
    }
    if !addrs[2].is_empty() {
        map.insert("current_dns_addr", addrs[2].to_string());
    }
    if !addrs[3].is_empty() {
        map.insert("mac_addr", addrs[3].to_string());
    }
    map.insert("interface", interface.to_string());

    Ok(map)
}

pub async fn wwan_info(con_name: &str) -> Result<HashMap<&'static str, String>, &'static str> {
    let args = &["wwan_info.sh", con_name];
    let output = match exec_command(args).await {
        Ok(o) => o,
        Err(_) => return Err("Could not get wwan info."),
    };
    let stdout = String::from_utf8_lossy(&output.stdout).trim().to_string();
    /*
        wwan_info[0]: APN
        wwan_info[1]: USER NAME
        wwan_info[2]: MCC/MNC
        wwan_info[3]: AUTH TYPE
        wwan_info[4]: IMSI
        wwan_info[5]: SIGNALQ
        wwan_info[6]: IMEI
        wwan_info[7]: BANDS
        wwan_info[8]: TAC
        wwan_info[9]: CELLID
    */
    let wwan_info = stdout
        .split(';')
        .collect::<Vec<_>>()
        .iter()
        .map(|s| s.trim().to_string())
        .collect::<Vec<String>>();

    let keys = [
        "apn",
        "user_name",
        "mccmnc",
        "auth",
        "imsi",
        "signalq",
        "imei",
        "bands",
        "tac",
        "cellid",
    ];
    let mut map = HashMap::new();
    for key in keys.iter() {
        map.insert(*key, "--".to_string());
    }
    for (i, info) in wwan_info.iter().enumerate() {
        if !info.is_empty() {
            map.insert(keys[i], info.to_string());
        }
    }
    Ok(map)
}
