// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    http::StatusCode,
    response::{Html, IntoResponse, Response},
};

use crate::common::{self, Config, SideMenu};

pub struct RawTemplate<T>(pub T);

impl<T> IntoResponse for RawTemplate<T>
where
    T: Template,
{
    fn into_response(self) -> Response {
        match self.0.render() {
            Ok(html) => Html(html).into_response(),
            Err(err) => (
                StatusCode::INTERNAL_SERVER_ERROR,
                format!("Failed to render template. Error: {err}"),
            )
                .into_response(),
        }
    }
}

#[derive(Template)]
#[template(path = "../src/common/templates/common.html")]
pub struct HtmlTemplate<T: Template> {
    side_menu: Vec<SideMenu>,
    title: String,
    header_title: String,
    model_number: String,
    version: &'static str,
    contents: T,
}

impl<T: Template> HtmlTemplate<T> {
    pub fn new<S: Into<String>>(title: S, contents: T) -> HtmlTemplate<T> {
        let customize = &Config::get().customize.get();
        HtmlTemplate {
            side_menu: customize.side_menu.clone(),
            title: title.into(),
            header_title: customize.header_title.clone(),
            model_number: customize.model_number.clone().unwrap_or_default(),
            version: common::VERSION,
            contents,
        }
    }
}

impl<T: Template> IntoResponse for HtmlTemplate<T> {
    fn into_response(self) -> Response {
        match self.render() {
            Ok(html) => Html(html).into_response(),
            Err(err) => (
                StatusCode::INTERNAL_SERVER_ERROR,
                format!("Failed to render template. Error: {err}"),
            )
                .into_response(),
        }
    }
}

// filters usable in askama if used in module
// https://askama.readthedocs.io/en/latest/filters.html#custom-filters
pub mod filters {
    use std::convert::Infallible;

    pub fn or_dash<'a, T>(
        value: &'a Option<T>,
        _: &'a dyn askama::Values,
    ) -> askama::Result<&'a str, Infallible>
    where
        T: AsRef<str>,
    {
        Ok(match value {
            Some(value) => value.as_ref(),
            None => "--",
        })
    }
}
