// SPDX-License-Identifier: MIT

use anyhow::anyhow;
use anyhow::Result;
use serde::{Deserialize, Serialize};

use crate::common::exec_command;

#[derive(Debug, Serialize)]
pub struct ContainerInfo {
    pub containers: Vec<Container>,
}

#[derive(Debug, Deserialize)]
#[serde(rename_all(deserialize = "PascalCase"))]
pub struct ContainerJson {
    pub id: Option<String>,
    pub names: Option<Vec<String>>,
    pub state: Option<String>,
    pub command: Option<Vec<String>>,
    pub image: Option<String>,
}

#[derive(Debug, PartialEq, Eq, Serialize)]
pub struct Container {
    pub id: String,
    pub name: String,
    pub state: String,
    pub command: Vec<String>,
    pub image: String,
}

impl ContainerInfo {
    pub fn from_podman_ps(output: &str) -> Result<Self> {
        let container_json: Vec<ContainerJson> = match serde_json::from_str(output) {
            Ok(v) => v,
            Err(e) => return Err(anyhow!("Could not parse container info json. {}", e)),
        };

        let mut containers = Vec::new();
        for c_obj in container_json {
            let id = c_obj.id.unwrap_or("".to_string());
            let names = c_obj.names.unwrap_or(vec!["".to_string()]);
            if id.is_empty() || names.is_empty() || names[0].is_empty() {
                continue;
            }

            let command = match c_obj.command {
                Some(cmd) => {
                    if cmd.is_empty() {
                        vec!["--".to_string()]
                    } else {
                        cmd
                    }
                }
                None => vec!["--".to_string()],
            };

            containers.push(Container {
                id,
                name: names[0].to_string(),
                state: c_obj.state.unwrap_or("--".to_string()),
                command,
                image: c_obj.image.unwrap_or("--".to_string()),
            });
        }
        containers.sort_unstable_by(|a, b| a.name.cmp(&b.name));
        Ok(ContainerInfo { containers })
    }

    pub async fn get() -> Result<Self> {
        let args = &["podman.sh", "ps", "-a", "--format=json"];
        ContainerInfo::from_podman_ps(&String::from_utf8_lossy(&exec_command(args).await?.stdout))
    }
}

#[cfg(test)]
mod tests {
    use crate::common::container::{Container, ContainerInfo};
    use anyhow::Result;
    #[test]
    fn test_parse_container() -> Result<()> {
        let info = ContainerInfo::from_podman_ps(
            r#"
[
    {
        "AutoRemove": false,
        "Command": [
            "sleep",
            "infinity"
        ],
        "CreatedAt": "16 hours ago",
        "Exited": false,
        "ExitedAt": -62135596800,
        "ExitCode": 0,
        "Id": "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5",
        "Image": "docker.io/library/alpine:latest",
        "ImageID": "5053b247d78b5e43b5543fec77c856ce70b8dc705d9f38336fa77736f25ff47c",
        "IsInfra": false,
        "Labels": null,
        "Mounts": [],
        "Names": [
            "gpio_example"
        ],
        "Namespaces": {
        },
        "Networks": [
            "podman"
        ],
        "Pid": 21671,
         "Pod": "",
        "PodName": "",
        "Ports": null,
        "Size": null,
        "StartedAt": 1688372015,
        "State": "running",
        "Status": "Up 16 hours ago",
        "Created": 1688372015
    },
    {
        "Command": [
            "sleep",
            "infinity"
        ],
        "Id": null,
        "Image": "docker.io/library/alpine:latest",
        "Names": [
            "gpio_example"
        ],
        "State": "running"
    },
    {
        "Command": [
            "sleep",
            "infinity"
        ],
        "Id": "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5",
        "Image": "docker.io/library/alpine:latest",
        "Names": null,
        "State": "running"
    },
    {
        "Command": [
            "sleep",
            "infinity"
        ],
        "Id": "",
        "Image": "docker.io/library/alpine:latest",
        "Names": [
            "gpio_example"
        ],
        "State": "running"
    },
    {
        "Command": [
            "sleep",
            "infinity"
        ],
        "Id": "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5",
        "Image": "docker.io/library/alpine:latest",
        "Names": [],
        "State": "running"
    },
    {
        "Command": [
            "sleep",
            "infinity"
        ],
        "Id": "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5",
        "Image": "docker.io/library/alpine:latest",
        "Names": [""],
        "State": "running"
    },
    {
        "Command": null,
        "Id": "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5",
        "Image": null,
        "Names": [
            "gpio_example"
        ],
        "State": null
    },
    {
        "Command": [],
        "Id": "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5",
        "Image": null,
        "Names": [
            "gpio_example"
        ],
        "State": null
    },
    {}
]
"#,
        )?;
        assert_eq!(info.containers.len(), 3);
        assert_eq!(
            info.containers[0],
            Container {
                name: "gpio_example".to_string(),
                id: "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5".to_string(),
                state: "running".to_string(),
                command: vec!["sleep".to_string(), "infinity".to_string()],
                image: "docker.io/library/alpine:latest".to_string(),
            }
        );
        assert_eq!(
            info.containers[1],
            Container {
                name: "gpio_example".to_string(),
                id: "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5".to_string(),
                state: "--".to_string(),
                command: vec!["--".to_string()],
                image: "--".to_string(),
            }
        );
        assert_eq!(
            info.containers[2],
            Container {
                name: "gpio_example".to_string(),
                id: "c278452f0dcd456e848c3c8e567eea048e469dc3c13a92fe960c362474ed35d5".to_string(),
                state: "--".to_string(),
                command: vec!["--".to_string()],
                image: "--".to_string(),
            }
        );
        Ok(())
    }
}
