// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use tokio::fs;
use tracing::warn;

use crate::common::exec_command_no_script;

#[derive(Clone, Debug)]
pub struct SwuConfig {
    pub initial_setup_installed: bool,
    pub watch_urls: Vec<String>,
}

impl SwuConfig {
    async fn check_initial_setup() -> Result<bool> {
        exec_command_no_script(&["abos-ctrl", "status", "--onetime-cert"])
            .await
            .context("Could not execute abos-ctrl status")
            .map(|_| true)
    }
    async fn get_watch_urls() -> Result<Vec<String>> {
        fs::read_to_string("/etc/swupdate.watch")
            .await
            .context("Could not read /etc/swupdate.watch")
            .map(|s| parse_watch_urls(&s))
    }
    pub async fn load() -> Self {
        let initial_setup_installed = SwuConfig::check_initial_setup().await.unwrap_or_else(|e| {
            warn!("checking for initial_setup status failed: {:?}", e);
            true
        });
        let watch_urls = SwuConfig::get_watch_urls().await.unwrap_or_else(|e| {
            warn!("getting watch urls failed: {:?}", e);
            vec![]
        });
        SwuConfig {
            initial_setup_installed,
            watch_urls,
        }
    }
}

/// just split by line and drop anything that doesn't start with http
fn parse_watch_urls(content: &str) -> Vec<String> {
    content
        .split('\n')
        .map(|s| s.trim())
        .filter(|s| s.starts_with("http"))
        .map(|s| s.to_string())
        .collect()
}

#[cfg(test)]
mod tests {
    use crate::common::config::swu::parse_watch_urls;

    #[test]
    fn test_parse_watch_urls() {
        let urls = parse_watch_urls(
            r#"# list of urls the swupdate-url service will regularily poll
# This file is not used unless the swupdate-url service has been enabled
 https://download.atmark-techno.com/armadillo-iot-g4/image/baseos-x2-latest.swu "#,
        );
        assert_eq!(
            &urls,
            &["https://download.atmark-techno.com/armadillo-iot-g4/image/baseos-x2-latest.swu"]
        );
        let urls = parse_watch_urls(
            r#"# list of urls the swupdate-url service will regularily poll
# This file is not used unless the swupdate-url service has been enabled
https://download.atmark-techno.com/armadillo-iot-a6e/image/baseos-6e-latest.swu
https://download.atmark-techno.com/armadillo-iot-a6e/image/a6e-gw-container-latest.swu"#,
        );
        assert_eq!(&urls, &[
                   "https://download.atmark-techno.com/armadillo-iot-a6e/image/baseos-6e-latest.swu",
                   "https://download.atmark-techno.com/armadillo-iot-a6e/image/a6e-gw-container-latest.swu"
        ]);
        let urls = parse_watch_urls(r#"# list of urls the swupdate-url"#);
        assert!(urls.is_empty());
    }
}
