// SPDX-License-Identifier: MIT

use anyhow::Result;
use once_cell::sync::OnceCell;
use std::sync::Arc;

mod customize;
mod hardware;
mod software;
mod swu;

pub use customize::Customize;
pub use customize::CustomizeConfig;
pub use customize::SideMenu;
pub use hardware::HardwareConfig;
pub use software::SoftwareConfig;
pub use swu::SwuConfig;

static CONFIG: OnceCell<Config> = OnceCell::new();

// Outwards-facing struct: wrap everything in an Arc/RwLock
// so it's safe to clone/update.
#[derive(Clone, Debug)]
pub struct Config {
    inner: Arc<ConfigInner>,
}

// (clone is only used for default config)
#[derive(Clone, Debug)]
pub struct ConfigInner {
    pub hardware: HardwareConfig,
    pub software: SoftwareConfig,
    pub swu: SwuConfig,
    pub customize: CustomizeConfig,
}

impl Config {
    async fn do_load() -> ConfigInner {
        let hardware = HardwareConfig::load().await;
        let swu = SwuConfig::load().await;
        let software = SoftwareConfig::load().await;
        let customize = CustomizeConfig::load().await;
        ConfigInner {
            hardware,
            software,
            swu,
            customize,
        }
    }

    pub async fn load() -> Result<()> {
        let inner = Config::do_load().await;
        CONFIG
            .set(Config {
                inner: inner.into(),
            })
            .expect("Load should only be called once");
        Ok(())
    }

    pub fn get() -> &'static ConfigInner {
        &CONFIG.get().expect("Config is not initialized").inner
    }
}
