// SPDX-License-Identifier: MIT

use gumdrop::Options;
use ipnet::IpNet;
use once_cell::sync::OnceCell;
use std::net::SocketAddr;
use tracing::Level;
use tracing_subscriber::prelude::*;

#[derive(Debug, Options)]
pub struct Args {
    #[options(
        no_short,
        parse(try_from_str = "split_subnets"),
        help = "space-separated subnets to allow connection from (default: local segments)"
    )]
    pub allowed_subnets: Vec<Vec<IpNet>>,

    #[options(no_short, help = "address/port to listen to", default = "[::]:58080")]
    pub listen_addr: SocketAddr,

    #[options(
        no_short,
        help = "Whether to require above address to be available at startup"
    )]
    pub free_bind: bool,

    // cannot use variables for default value, so define these twice to keep
    // existing debug build path
    #[cfg(debug_assertions)]
    #[options(no_short, help = "path to TLS key", default = "tls/key.pem")]
    pub tls_key: String,
    #[cfg(debug_assertions)]
    #[options(no_short, help = "path to TLS certificate", default = "tls/cert.pem")]
    pub tls_cert: String,

    #[cfg(not(debug_assertions))]
    #[options(
        no_short,
        help = "path to TLS key",
        default = "/etc/atmark/abos_web/tls/key.pem"
    )]
    pub tls_key: String,
    #[cfg(not(debug_assertions))]
    #[options(
        no_short,
        help = "path to TLS certificate",
        default = "/etc/atmark/abos_web/tls/cert.pem"
    )]
    pub tls_cert: String,

    #[options(
        no_short,
        help = "User to drop privileges to",
        default = "abos-web-admin"
    )]
    pub user: String,

    // up to 4 times only increment abos_web traces
    // more increment all traces for all crates
    #[options(count, help = "Increase verbosity (up to 7 times)")]
    verbose: u8,

    #[options(help = "print help message")]
    help: bool,
}

fn split_subnets(s: &str) -> Result<Vec<IpNet>, ipnet::AddrParseError> {
    s.split(' ').map(|x| x.parse()).collect()
}

static ARGS: OnceCell<Args> = OnceCell::new();

fn init_logs() {
    let args = args();
    let abos_web_level = match args.verbose {
        0 => Level::WARN,
        1 => Level::INFO,
        2 => Level::DEBUG,
        _ => Level::TRACE,
    };
    let default_level = match args.verbose {
        i if i <= 3 => Level::ERROR,
        4 => Level::WARN,
        5 => Level::INFO,
        6 => Level::DEBUG,
        _ => Level::TRACE,
    };
    let filter = tracing_subscriber::filter::Targets::new()
        .with_target("abos_web", abos_web_level)
        .with_default(default_level);
    tracing_subscriber::registry()
        .with(tracing_subscriber::fmt::layer())
        .with(filter)
        .init();
}

pub fn args_init() {
    ARGS.set(Args::parse_args_default_or_exit())
        .expect("args init should only be called once");

    init_logs()
}

pub fn args() -> &'static Args {
    ARGS.get().expect("args init should have been called")
}
