#!/bin/sh -e
# SPDX-License-Identifier: MIT

BRIDGE_NAME="$1"
BRIDGE_ADDR="$2"

HW_MODE="$3"
CHANNEL="$4"
SSID=$(echo "$5" | sed -e 's:[\\/&]:\\&:g')
WPA_PASSPHRASE=$(echo "$6" | sed -e 's:[\\/&]:\\&:g')
INTERFACE="$7"

BRIDGE_CON_FILE="/etc/NetworkManager/system-connections/$BRIDGE_NAME.nmconnection"
NM_IGNORE_IF_CONF="/etc/NetworkManager/conf.d/90_abosweb_disable_wlan.conf"
HOSTAPD_CONF="/etc/hostapd/hostapd.conf"
BRIDGE_IF="br_ap"

# fail early if iface not found
ip link set up dev "$INTERFACE"

trap catch ERR

catch() {
	local rc="$?"
	set +e

	persist_file -R "$BRIDGE_CON_FILE" \
		"$NM_IGNORE_IF_CONF" \
		/etc/runlevels/default/hostapd \
		"$HOSTAPD_CONF"

	if ! [ -e "$BRIDGE_CON_FILE" ]; then
		nmcli con del "$BRIDGE_NAME" >/dev/null 2>&1
	else
		nmcli con load "$BRIDGE_CON_FILE"
		nmcli con up "$BRIDGE_NAME"
	fi

	if ! [ -e "$NM_IGNORE_IF_CONF" ] && [ -n "$NMCLI_IGNORE_IF" ]; then
		nmcli d set "$NMCLI_IGNORE_IF" managed yes
		nmcli d wifi list --rescan yes
	fi

	if [ -e "/etc/runlevels/default/hostapd" ]; then
		rc-service hostapd restart
	fi

	rm -f "$HOSTAPD_CONF.abos_web"

	echo "Could not set up Access point." >&2
	exit $rc
}

if ! nmcli con show "$BRIDGE_NAME" >/dev/null 2>&1; then
	nmcli con add con-name "$BRIDGE_NAME" type bridge ifname "$BRIDGE_IF"
	nmcli con mod "$BRIDGE_NAME" ipv4.method manual ipv4.address "$BRIDGE_ADDR" bridge.stp off
else
	nmcli con mod "$BRIDGE_NAME" ifname "$BRIDGE_IF" ipv4.method manual ipv4.address "$BRIDGE_ADDR" bridge.stp off
fi
nmcli con up "$BRIDGE_NAME"

# disable nmcli side
NMCLI_IGNORE_IF="$INTERFACE"
if [ "$INTERFACE" = uap0 ] && [ -e /sys/class/net/mlan0 ]; then
	NMCLI_IGNORE_IF=mlan0
fi
cat > "$NM_IGNORE_IF_CONF" <<EOF
[device_abosweb_$NMCLI_IGNORE_IF]
match-device=interface-name:$NMCLI_IGNORE_IF
managed=0
EOF
nmcli d set "$NMCLI_IGNORE_IF" managed no
ip addr flush dev "$NMCLI_IGNORE_IF"
if [ "$INTERFACE" != "$NMCLI_IGNORE_IF" ]; then
	ip link set down dev "$NMCLI_IGNORE_IF"
fi

# make file not readable by non-root
umask 0077
sed -e "s/^hw_mode=.*/hw_mode=$HW_MODE/" \
	-e "s/^channel=.*/channel=$CHANNEL/" \
	-e "s/^ssid=.*/ssid=$SSID/" \
	-e "s/^wpa_passphrase=.*/wpa_passphrase=$WPA_PASSPHRASE/" \
	-e "s/^interface=.*/interface=$INTERFACE/" \
	-e "s/^bridge=.*/bridge=$BRIDGE_IF/" \
	"$HOSTAPD_CONF".example > "$HOSTAPD_CONF.abos_web"

# hostapd does not have a check conf mode, run it in background mode and
# if that did not error the file is probably ok...
# (we need to stop service first if modifying ap config)
rc-service hostapd stop > /dev/null 2>&1 || :
hostapd -B "$HOSTAPD_CONF.abos_web"

pkill hostapd
mv "$HOSTAPD_CONF.abos_web" "$HOSTAPD_CONF"


rc-service hostapd restart
rc-update add hostapd

persist_file "$BRIDGE_CON_FILE" \
	"$NM_IGNORE_IF_CONF" \
	/etc/runlevels/default/hostapd \
	"$HOSTAPD_CONF"
