#!/bin/sh
# SPDX-License-Identifier: MIT

set -xe

error() {
	printf "%s\n" "$@" >&2
	exit 1
}

case "$1" in
-*|"") error "Usage: $0 hostname [nobuild|noinst|norestart|notls|sshconf <conf>|norestapi|variant-set <name>]";;
esac

cd "$(dirname "$0")/.."

NOBUILD=""
NOINST=""
NORESTART=""
ssh_conf=""
build_mode=--release
rustls="rustls"
restapi="restapi"
variant_set=""
target="$1"
shift

while [ -n "$1" ]; do
	case "$1" in
	nobuild) NOBUILD=1;;
	noinst) NOINST=1;;
	norestart) NORESTART=1;;
	notls) rustls="";;
	norestapi) restapi="";;
	sshconf) ssh_conf="$2"; shift;;
	variant-set) variant_set="$2"; shift;;
	*) error "Unknown arg $1";;
	esac
	shift
done

arch=$(ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" uname -m) || error "Could not ssh to $target"
case "$arch" in
aarch64)
	arch=aarch64-unknown-linux-musl
	;;
armv7l)
	arch=armv7-unknown-linux-musleabi
	;;
*)
	error "Only support aarch64 or armv7l: $arch"
	;;
esac

if [ -z "$NOBUILD" ]; then
	# cross isn't compatible with mold despite specifying x86_64 target..
	moved_cargo_conf=""
	if [ -e ~/.cargo/config.toml ]; then
		mv ~/.cargo/config.toml ~/.cargo/config.toml.old
		trap "mv ~/.cargo/config.toml.old ~/.cargo/config.toml" EXIT INT QUIT TERM
		moved_cargo_conf=1
	fi
	cross build --target="$arch" "$build_mode" \
		--no-default-features \
		${rustls:+--features tls-rustls} \
		${restapi:+--features restapi}
	if [ -n "$moved_cargo_conf" ]; then
		mv ~/.cargo/config.toml.old ~/.cargo/config.toml
		trap - EXIT INT QUIT TERM
	fi
fi
[ -z "$NOINST" ] || exit 0

tar --owner=root -c etc | ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" tar -C / -xv
tar --owner=root -c variant | ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" tar -C /usr/share/abos-web -xv
tar --owner=root -h -c static | ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" tar -C /usr/share/abos-web -xv
tar --owner=root -C scripts -c . | ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" tar -C /usr/libexec/abos-web -xv

if [ -n "$variant_set" ]; then
	ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" 'echo "ABOSWEB_VARIANT=\"'"$variant_set"'\"" >> /etc/conf.d/abos-web'
fi

if [ -z "$NORESTART" ]; then
	ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" 'rc-service abos-web stop; pkill abos-web || true'
	# cannot copy binary if service is running
	scp  ${ssh_conf:+-F "$ssh_conf"} "target/$arch/release/abos-web" "$target":/usr/bin
fi

ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" 'chown -R abos-web-admin: /etc/atmark/abos_web \
	&& chmod 644 /etc/doas.d/abos_web.conf \
	&& persist_file -rv /etc/atmark/abos_web /etc/doas.d /etc/init.d/abos-web /usr/libexec/abos-web /usr/share/abos-web /usr/bin/abos-web'

if [ -z "$NORESTART" ]; then
	ssh  ${ssh_conf:+-F "$ssh_conf"} "$target" rc-service abos-web start
fi
