#!/bin/sh
# SPDX-License-Identifier: MIT

# This scripts install a local cargo toolchain in the test directory,
# builds and install abos-web on the target machine and then
# runs tests in the official playwright container
#
# For developement, it's recommended to install dependencies directly
# and run either utils/update_armadillo.sh for build/update and
# tests/run_playwright.sh to run a locally installed version of
# this test.

set -x

error() {
        printf "%s\n" "$@" >&2
        exit 1
}

cleanup() {
	# remove test artifacts from previous runs
	rm -rf playwright/screenshots playwright/test-results
}

setup() {
	export RUSTUP_HOME="$PWD/bins/rustup"
	export CARGO_HOME="$PWD/bins/cargo"
	export PATH="$PWD/bins/cargo/bin:$PATH"
	if command -v jq >/dev/null; then
		PLAYWRIGHT_VERSION=$(jq -r '.packages."node_modules/@playwright/test".version' < playwright/package-lock.json)
	else
		# ugly grep...
		PLAYWRIGHT_VERSION=$(grep -A 1 'node_modules/@playwright/test' playwright/package-lock.json \
					| sed -ne 's/.*version": "\([^"]*\)".*/\1/p')
	fi
	[ -n "$PLAYWRIGHT_VERSION" ] || error "Could not get playwright version from playwright/package-lock.json"
	PLAYWRIGHT_CTR="mcr.microsoft.com/playwright:v$PLAYWRIGHT_VERSION-jammy"

	if ! docker image inspect "$PLAYWRIGHT_CTR" >/dev/null 2>&1; then
		docker pull "$PLAYWRIGHT_CTR" || exit
	fi


	if ! [ -e bins/rustup ]; then
		# cross cannot run in a container because it needs to run its own container...
		# so we install our own rustup and everything in workspace
		mkdir -p bins/xargo
		curl https://sh.rustup.rs -sSf | sh -s -- --profile minimal --no-modify-path --target aarch64-unknown-linux-musl -y || exit
	fi
	if ! [ -e bins/cargo/bin/cross ]; then
		cargo install cross
	fi

	case "$armadillo" in
	*%*%*) error "MUltiple % signs not handled";;
	*%*)
		# double % for ssh config
		armadillo="${armadillo%\%*}%%${armadillo#*%}"
		;;
	esac
	cat > bins/ssh_config <<EOF
Host armadillo
	Hostname $armadillo
	User root
	BatchMode yes
	UserKnownHostsFile "$PWD/bins/ssh_known_hosts"
	StrictHostKeyChecking accept-new
EOF
	rm -f bins/ssh_known_hosts

	# retry a few times to ssh if armadillo was freed up during reboot
	local retries=0
	while ! ssh -F bins/ssh_config armadillo true; do
		[ "$retries" -lt 3 ] || error "Could not ssh into $armadillo"
		retries=$((retries+1))
		sleep 5;
	done

	case "$armadillo" in
	fe80*)
		# browsers can't use local-link addresses yet... get ipv4.
		armadillo=$(ssh -F bins/ssh_config armadillo ip -4 a show dev eth0 \
			| sed -ne 's/.*inet \([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*\).*/\1/p')
		[ -n "$armadillo" ] || error "Could not resolve address for armadillo"
		;;
	esac
}

update_armadillo() {
	echo 'abos-web-admin:abosweb_autotest' | ssh -F "$PWD/bins/ssh_config" armadillo chpasswd || exit

	case "$TEST_MODE" in
	apk)
		# ensure latest package is installed/in use and restart service
		# 'apk fix' only reinstalls if same version is latest,
		# 'apk fix -u' only reinstalls if an upgrade is available,
		# so run both in that order to reinstall exactly once...
		ssh -F "$PWD/bins/ssh_config" armadillo '
				apk fix abos-web && apk fix -u abos-web \
					&& rc-service abos-web restart' \
			|| exit
		;;
	*)
		# by default build & install update from sources
		../utils/update_armadillo.sh armadillo \
				sshconf "$PWD/bins/ssh_config" \
				${NO_RUN_RESTAPI:+norestapi} \
			|| exit
		;;
	esac
}

prepare_armadillo() {
	ssh -F "$PWD/bins/ssh_config" armadillo "podman kill -a \
		&& podman rm -a -f \
		&& podman run -d --init --name alpine docker.io/alpine \
			sh -c 'echo ok; exec sleep infinity'" || exit
}

run_tests() {
	cat > bins/.env <<EOF
ABOSWEB_BASE=https://$armadillo:58080
ABOSWEB_PASSWORD=abosweb_autotest
ABOSWEB_SSH_CONFIG="$PWD/bins/ssh_config"
CI=1
HOME=/tests
EOF
	cat > bins/.run.sh <<'EOF'
# chromium is bugged and picks a chinese font despite lang=ja if font is available
apt purge -y fonts-wqy-zenhei fonts-tlwg-loma-otf
# run tests as user appropriate for playwright directory
# (podman user mode: root, docker: whatever owns the 'playwright' directory)
who=$(stat -c %u:%g playwright)
if [ "$who" = 0:0 ]; then
	cd playwright || exit
	npm ci || exit
	exec npx playwright test
fi
addgroup --system --gid ${who#*:} testgroup || exit
adduser --system --uid ${who%:*} --ingroup testgroup \
	--home /tests/playwright --no-create-home \
	--shell /bin/bash testuser || exit
su testuser -c 'cd && npm ci && npx playwright test'
EOF
	# run rest api curl tests -- we run them first to prepare settings for manual if required
	if [ -z "$NO_RUN_RESTAPI" ]; then
		./run_curl.sh bins/.env || exit
	fi
	# run playwright
	docker run --rm \
			-v "$PWD:/tests" -w /tests \
			--env-file "$PWD/bins/.env" \
			"$PLAYWRIGHT_CTR" \
			sh /tests/bins/.run.sh \
		|| exit
}

main() {
	[ "$#" = 1 ] && [ "$1" != "-h" ] || error "Usage: $0 <armadillo_address>"
	armadillo="$1"

	cd "$(dirname "$0")" || exit
	cleanup
	setup
	update_armadillo
	prepare_armadillo
	run_tests
}

main "$@"
