// SPDX-License-Identifier: MIT

import { test, expect } from '@playwright/test';
import { screenshot } from '../lib/screenshot.ts';

test('admin setting page test', async ({ page }) => {
	await page.goto('/time');

	await expect(page).toHaveTitle(/時刻設定/);

	// manual screenshot (must run first, assuming rest API ran first and sync'd things)
	await test.step('Manual screenshots', async () => {
		await screenshot(page, {
			path: 'screenshots/manual/time_status_sync.png',
			locator: '#time_status_view',
			margin: 5,
		});
		await screenshot(page, {
			path: 'screenshots/manual/time_ntp_config.png',
			locator: '#ntp_config_view',
			margin: 5,
		});
		await screenshot(page, {
			path: 'screenshots/manual/time_zone_config.png',
			locator: '#time_zone_view',
			margin: 5,
		});
	});

	await test.step('Change ntp settings', async () => {
		// remove initstepslew
		await page.locator('#initstepslew').fill('');
		// add second server
		await page.getByRole('button', { name: 'サーバー追加' }).click();
		await page.locator('#ntp_server_1').click();
		await page.locator('#ntp_server_1').fill('server 127.0.0.1');
		// submit
		await page.locator('#ntp_config_view').getByRole('button', { name: '設定' }).click();
		// check we got what we wanted
		await expect(page.locator('#ntp_server_1')).toHaveValue('server 127.0.0.1');
		await expect(page.locator('#initstepslew')).toHaveValue('');

	});

	await test.step('Manual screenshots (unsync)', async() => {
		await screenshot(page, {
			path: 'screenshots/manual/time_status_unsync.png',
			locator: '#time_status_view',
			margin: 5,
		});
	});

	await test.step('Restore ntp settings', async() => {
		page.once('dialog', dialog => {
			dialog.accept().catch(() => {});
		});
		await page.getByRole('button', { name: 'デフォルトに戻す' }).click();
		await expect(page.locator('#initstepslew')).toHaveValue('10 pool.ntp.org');
		await expect(page.locator('#ntp_server_1')).toHaveCount(0);
	});

	await test.step('Change timezone', async() => {
		// This also checks that the option exists
		await page.getByLabel('タイムゾーン').selectOption('Europe/Paris');
		await page.locator('#time_zone_view').getByRole('button', { name: '設定' }).click();
		// France is UTC +1 or 2 depending on DST
		// FIXME: for some reason the content here does not seem to always
		// be updated and fails ~1/4 of the time despite adding extra wait
		// or page loads, and working when manually opening the page after
		// failure. Some cache effect?
		// For now we mostly care about screenshots to skip this bit.
		// await expect(page.locator('#timestamp_now')).toContainText(/ \+0[12]:00/);

		await page.getByLabel('タイムゾーン').selectOption('Asia/Tokyo');
		await page.locator('#time_zone_view').getByRole('button', { name: '設定' }).click();
		// FIXME
		// await expect(page.locator('#timestamp_now')).toContainText(' +09:00');
	});
});

