// SPDX-License-Identifier: MIT

import { Page, expect } from '@playwright/test';

// helper to frame screenshot with a margin around locator
// Might replace it with something better depending on playwright
// changes: https://github.com/microsoft/playwright/issues/28394
async function get_target(
	page: Page,
	locator?: string,
	margin?: number
): [Page | Locator, null | { x: number, y: number, width: number, height: number }] {
	if (!margin) {
		if (locator) {
			var target = page.locator(locator);
			await target.scrollIntoViewIfNeeded();
			return [ target, null ];
		}
		return [ page, null ];
	}
	if (!locator) {
		throw "Must have a locator if margin was set";
	}
	var target = page.locator(locator);
	await target.scrollIntoViewIfNeeded();
	var box = await target.boundingBox();
	const page_height = page.viewportSize().height;
	if (box.y + box.height + margin > page_height) {
		// need to scroll a bit more for the margin...
		// Use 2 margins for safety
		await page.evaluate((margin) => {
			window.scrollBy(0, 2 * margin);
		}, margin);
		box = await target.boundingBox();
	}
	if (box.y - margin < 0 || box.y + box.height + margin > page_height) {
		throw "item is too big to fit view port, resize viewport";
	}
	return [ page, {
		x: box.x - margin,
		y: box.y - margin,
		width: box.width + 2 * margin,
		height: box.height + 2 * margin,
	}];
}

export async function screenshot(
	page: Page,
	opts: {
		// path to store screenshot
		path: string,
		// page locator for partial screenshot
		locator?: string,
		// extra styles if required // playwight 1.42+
		style?: string,
		// disable background? keep by default
		omitBackground?: boolean,
		// mask items e.g. [page.locator('foo')]
		mask?: Array<Locator>,
		// margin around element if set
		margin?: number
	},
) {
	var style = opts.style || "";
	var omitBackground = opts.omitBackground || false;
	if (omitBackground) {
		// need plain background to make background actually transparent
		style += '\nbody { background: none; }';
		// firefox doesn't support omitBackground screenshot property as of playwright 1.41...
		if (page.context().browser().browserType().name() == 'firefox') {
		       	omitBackground = false;
		}
	}

	// get screenshot box area
	var [target, clip] = await get_target(page, opts.locator, opts.margin);


	return await target.screenshot({
		path: opts.path,
		clip: clip,
		omitBackground: omitBackground,
		style: style,
		mask: opts.mask,
		// override default pink
		maskColor: "#222222",
	});
}
