#!/bin/sh
# SPDX-License-Identifier: MIT

. "$(dirname "$0")/common"

[ -n "$ABOSWEB_PASSWORD" ] || error "ABOSWEB_PASSWORD must be set"

new_token_by_cookie() {
	# test token creation through logged in session
	local token

	req "/login" -c cookie -d "password=$ABOSWEB_PASSWORD" \
		|| error "Could not login"
	req "/api/tokens" -b cookie -X POST \
		|| error "Could not get a new token by cookie"
	token=$(jq -r '.token'< req) || error "token creation had no token"
	check_req_jq default_admin '.permissions[0]' Admin
	req_bearer "/api/tokens/$token" -X DELETE \
		|| error "Could not delete new token"
}

new_token_json_permissions() {
	# this also tests token creation by token
	local token

	req_json "/api/tokens" -X POST -d '{"permissions": ["ContainerAdmin"]}' \
		|| error "Could not create token with permission (json)"
	token=$(jq -r '.token'< req) || error "token creation had no token"
	check_req_jq permission '.permissions[0]' ContainerAdmin

	req_bearer "/api/tokens" \
		|| error "Could not list tokens"
	[ "$(jq -r ".tokens[] | select(.token == \"$token\") | .permissions[]" < req)" = "ContainerAdmin" ] \
		|| error "Token not listed or incorrect permission in list"

	req_bearer "/api/tokens/$token" -X DELETE \
		|| error "Could not delete new token"
}

new_token_form_permissions() {
	local token

	req_bearer "/api/tokens" -X POST -d 'permissions=SwuInstall' -d "permissions=ContainerView" \
		|| error "Could not create token with permission (form)"
	token=$(jq -r '.token'< req) || error "token creation had no token"
	check_req_jq permission_0 '.permissions[0]' SwuInstall
	check_req_jq permission_1 '.permissions[1]' ContainerView

	# also test getting an individual token vs. getting full list in json_permissions
	req_bearer "/api/tokens/$token" \
		|| error "Could not list tokens"
	check_req_jq permissions '.permissions | sort[]' "ContainerView
SwuInstall"

	req_bearer "/api/tokens/$token" -X POST -d "permissions=ContainerView" \
		|| error "Could not update token with single permission (form)"

	req_bearer "/api/tokens/$token" \
		|| error "Could not list tokens"
	check_req_jq permission_change '.permissions[]' ContainerView

	# check it's usable
	req "/api/containers" -H "Authorization: Bearer $token" \
		|| error "Could not list containers with ContainerView token"

	req_bearer "/api/tokens/$token" -X DELETE \
		|| error "Could not delete new token"
}

mod_token() {
	local token
	token="$(cat /proc/sys/kernel/random/uuid)" \
		|| skip "Could not get random id"

	# mod with a new path should create it
	req_bearer "/api/tokens/$token" -X POST \
		|| error "Could not create token by name"

	req_bearer "/api/tokens/$token" \
		|| error "Token $token was not really created?"
	check_req_jq new_admin '.permissions[]' Admin

	req_bearer "/api/tokens/$token" -X POST \
			-d "permissions=ContainerView" -d "permissions=SwuInstall" \
		|| error "Could not update $token token"

	req_bearer "/api/tokens/$token" \
		|| error "Could not get $token again"
	check_req_jq mod_form '.permissions[]' "ContainerView
SwuInstall"

	req_bearer "/api/tokens/$token" -X DELETE \
		|| error "Could not delete new token"
}

run_tests new_token_by_cookie new_token_json_permissions \
	new_token_form_permissions mod_token
