#!/bin/sh
# SPDX-License-Identifier: MIT

. "$(dirname "$0")/common"

install_swu_url() {
	req_bearer "/api/swu/install/url" -d url=https://localhost:58080/fail \
		|| error "Could not install by url"

	grep -qF "SWUPDATE failed [0] ERROR : Channel operation returned HTTP error code 404" req \
		|| error "Did not get 404 error from swupdate"
	check_req_jq exit_code_url '.[-1].exit_code' 1 -r -s
}

install_swu_upload() {
	req_bearer "/api/swu/install/upload" -F swu=@"$TESTDIR/data/swu/empty_badsig.swu" \
		|| error "Could not install by upload"
	grep -qE "SWUPDATE failed \[0\] ERROR : .*sw-description\.sig cannot be parsed as DER-encoded CMS signature blob" req \
		|| error "Did not get invalid signature error from swupdate"
	check_req_jq exit_code_upload '.[-1].exit_code' 1 -r -s
}

view_versions() {
	local abos_version json_version

	req_bearer "/api/swu/versions" \
		|| error "Could not get sw-versions"
	json_version=$(jq -r '.base_os // ""' < req)
	[ -n "$json_version" ] || error "Versions did not contain base_os version?"

	if abos_version=$(ssh_armadillo "awk '\$1 == \"base_os\" { print \$2 }' /etc/sw-versions"); then
		[ "$abos_version" = "$json_version" ] \
			|| error "/etc/atmark-release version $abos_version did not match rest version $json_version"
	fi
}

view_status() {
	local abos_status abos_rollback=true rollback date versions

	abos_status=$(ssh_armadillo abos-ctrl status --rollback -vv) \
		|| abos_rollback=false
	[ -n "$abos_status" ] || skip "Fresh install? Skipping view status check"

	req_bearer "/api/swu/status" \
		|| error "Could not get swu status"
	rollback=$(jq -r '.rollback_ok // ""' < req)
	[ -n "$rollback" ] || error "Status did not contain rollback_ok?"

	date=$(jq -r '.last_update_timestamp // ""' < req)
	[ -n "$date" ] || error "Status did not contain last_update_timestamp?"
	# busybox date output isn't the same as gnu date's... try to normalize
	date=$(date -d "@$date" +"%a %b %_d %H:%M:%S %Z")

	# reconstruct abos-ctrl status output from dict...
	versions=$(jq -r  '.last_update_versions | to_entries[] | "  \(.key): \(.value[0] // "unset") -> \(.value[1] // "unset")"' < req)

	[ "$rollback" = "$abos_rollback" ] \
		|| error "rollback value seems wrong (abos-web said $rollback, but abos-ctrl status --rollback command was $abos_rollback)"
	echo "$abos_status" | grep -qF "$date" \
		|| error "Date didn't match, abos-web had $date:" "$abos_status"
	echo "$abos_status" | grep -qF "$versions" \
		|| error "Versions didn't match, abos-web had:" "$versions" \
			"abos-ctrl status had:" "$abos_status"
}

run_tests install_swu_url install_swu_upload || exit

if ! ssh_armadillo true; then
	echo "Need to be able to ssh into 'armadillo' for other tests"
	exit
fi

run_tests view_versions view_status
