// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use axum::{
    response::IntoResponse,
    routing::{delete, get, post},
    Router,
};
use regex::bytes::Regex;
use serde::{Deserialize, Serialize};
use serde_json::json;

use crate::common::{
    self, json_response, networkmanager::parse_nmcli_t_line, CheckAuthRestApi, JsonOrForm,
    JsonOrFormOption, RestApiPermissionNetworkAdmin, RestApiPermissionNetworkView,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/wlan/scan", get(rest_wlan_scan))
        .route("/api/wlan/connect", post(rest_wlan_connect))
        .route("/api/wlan/ap", post(rest_ap_setup))
        .route("/api/wlan/ap", delete(rest_ap_delete))
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct ScanParam {
    rescan: Option<bool>,
}

#[derive(Serialize, Debug)]
struct RestWlanSSID {
    id: String,
    signal: u8,
    bssid: String,
    chan: u8,
    rate: String,
    security: String,
    active: String,
}

async fn rest_get_ssid_list(rescan: bool) -> Result<Vec<RestWlanSSID>> {
    let mut args = vec!["get_ssid_list.sh"];
    if !rescan {
        args.push("--no-rescan");
    }
    let output = common::exec_command(&args).await?;
    let stdout = String::from_utf8_lossy(&output.stdout);
    Ok(stdout
        .lines()
        .map(parse_nmcli_t_line)
        .filter_map(|a| {
            // field order as per get_ssid_list.sh:
            // SSID,SIGNAL,BSSID,CHAN,RATE,SECURITY,ACTIVE
            Some(RestWlanSSID {
                id: a.first()?.to_string(),
                signal: a.get(1)?.parse().ok()?,
                bssid: a.get(2)?.to_string(),
                chan: a.get(3)?.parse().ok()?,
                rate: a.get(4)?.to_string(),
                security: a.get(5)?.to_string(),
                active: a.get(6)?.to_string(),
            })
        })
        .collect())
}

/// GET "/api/wlan/scan"
/// - Access: Network view
/// - Input: rescan: optional bool (default true)
/// - Output: List of objects {id, signal, bssid, chan, rate, security, active}
async fn rest_wlan_scan(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
    JsonOrFormOption(params): JsonOrFormOption<ScanParam>,
) -> ErrorStringResult {
    let mut rescan = true;
    if let Some(params) = params {
        rescan = params.rescan.unwrap_or(true);
    }
    let scan_result = rest_get_ssid_list(rescan).await?;
    json_response(&scan_result)
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct ConnectParam {
    ssid: String,
    passphrase: Option<String>,
    interface: Option<String>,
    bssid: Option<String>,
    hidden: Option<bool>,
}

// connect output is something like this on success (including reconnects,
// which re-use the same id)
//   Device 'mlan0' successfully activated with '178b8c95-fcad-4bb1-8040-5a02b9ad046f'.
const CONNECT_RE: &str = r#"activated with '([^']*)'"#;

/// POST "/api/wlan/connect"
/// - Access: Network admin
/// - Input: ssid, passphrase, interface, bssid, hidden (bool).
///   Everything except ssid is optional.
/// - Output: object with uuid, new connection uuid
async fn rest_wlan_connect(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    JsonOrForm(params): JsonOrForm<ConnectParam>,
) -> ErrorStringResult {
    let mut args = vec!["nmcli.sh", "wifi_connect", &params.ssid];
    if let Some(passphrase) = &params.passphrase {
        args.push("password");
        args.push(passphrase);
    }
    if let Some(interface) = &params.interface {
        args.push("ifname");
        args.push(interface);
    }
    if let Some(bssid) = &params.bssid {
        args.push("bssid");
        args.push(bssid);
    }
    if let Some(hidden) = &params.hidden {
        args.push("hidden");
        args.push(hidden.then_some("yes").unwrap_or("no"));
    }
    // build RE before connecting for error handling
    let re = Regex::new(CONNECT_RE).context("Could not build regex?!")?;

    let output = common::exec_command(&args).await?;
    let uuid = re
        .captures(&output.stdout)
        .and_then(|c| c.get(1))
        .context("Connected, but could not get uuid")?;

    json_response(&json!({
        "uuid": String::from_utf8_lossy(uuid.as_bytes()),
    }))
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct ApSetupParam {
    ssid: String,
    passphrase: String,
    bridge_addr: String,
    hw_mode: Option<String>,
    channel: Option<u32>,
    interface: Option<String>,
}

/// POST "/api/wlan/ap"
/// - Access: Network admin
/// - Input: ssid, passphrase, bridge_addr, hw_mode, channel, interface
///   interface is optional. if hw_mode is not set it is computed from channel.
///   if channel is not set it is automatically set within hw_mode frequencies.
/// - Output: None
async fn rest_ap_setup(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    JsonOrForm(params): JsonOrForm<ApSetupParam>,
) -> ErrorStringResult {
    // clippy false positive
    // https://github.com/rust-lang/rust-clippy/issues/13073
    #[allow(clippy::redundant_closure)]
    let interface = params
        .interface
        .as_deref()
        .or_else(|| common::ap_interface())
        .context("Could not find wlan network interface.")?;

    // channel=0 is ACS survey, which will scan for a while and select
    // the least busy channel automatically
    let channel = params.channel.unwrap_or(0);
    // channel 1-14 is g, 35+ is a; if it wasn't set error out.
    let hw_mode = match params.hw_mode.as_deref() {
        Some(mode) => mode,
        None if channel == 0 => Err("At least either hw_mode or channel must be set")?,
        None if channel < 20 => "g",
        None => "a",
    };
    let channel_str = channel.to_string();

    let args = &[
        "ap_setup.sh",
        common::AP_BRIDGE_NAME,
        &params.bridge_addr,
        hw_mode,
        &channel_str,
        &params.ssid,
        &params.passphrase,
        interface,
    ];
    common::exec_command(args).await?;

    Ok(().into_response())
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct ApDeleteParam {
    interface: Option<String>,
}

/// DELETE "/api/wlan/ap"
/// - Access: Network admin
/// - Input: interface (optional)
/// - Output: None
async fn rest_ap_delete(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    JsonOrFormOption(params): JsonOrFormOption<ApDeleteParam>,
) -> ErrorStringResult {
    // same clippy bug
    #[allow(clippy::redundant_closure)]
    let interface = params
        .as_ref()
        .and_then(|p| p.interface.as_deref())
        .or_else(|| common::ap_interface())
        .context("Could not find wlan network interface.")?;

    let args = &["ap_delete.sh", common::AP_BRIDGE_NAME, interface];
    common::exec_command(args).await?;

    Ok(().into_response())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_extract_id_regex() {
        let re = Regex::new(CONNECT_RE).unwrap();
        let output =
            b"Device 'mlan0' successfully activated with '178b8c95-fcad-4bb1-8040-5a02b9ad046f'.";
        let uuid = re.captures(output).and_then(|c| c.get(1)).unwrap();
        assert_eq!(uuid.as_bytes(), b"178b8c95-fcad-4bb1-8040-5a02b9ad046f");
    }
}
