// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use askama::Template;
use axum::{
    extract::{Extension, Form},
    http::StatusCode,
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use futures::future::join_all;
use ipnet::Ipv4Net;
use serde::{Deserialize, Serialize};
use std::{collections::HashSet, net::Ipv4Addr, str::FromStr};
use tower_sessions::Session;

use crate::common::{
    self, check_auth, filters, get_title,
    networkmanager::{nmcon_act, nmconvec, parse_nmcli_t_line, NMCon},
    Config, HtmlTemplate, LoggedIn, Title,
};
use crate::error::{ErrorStringResult, PageResult};

mod hostapd;

use hostapd::HostapdConfig;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize)]
struct WlanParam {
    essid: String,
    password: String,
    security: String,
    method: String,
    ip_addr: String,
    netmask: String,
    gateway_addr: String,
    p_dns_addr: String,
    s_dns_addr: String,
}

#[derive(Deserialize, Debug)]
pub struct WlanSaved {
    pub uuid: String,
    pub ssid: String,
    pub autoconnect: String,
}

#[derive(Default)]
pub struct StaInfo {
    pub uuid: Option<String>,
    pub ssid: Option<String>,
    pub security: Option<String>,
    pub ip_addr: Option<String>,
    pub gw_addr: Option<String>,
    pub interface: Option<String>,
    pub mac_addr: Option<String>,
}

pub struct WlanInfo {
    pub sta: StaInfo,
    pub wlan_saved: Vec<WlanSaved>,
    pub hostapd: HostapdConfig,
    pub bridge_addr: Option<String>,
}

#[derive(Serialize, Debug)]
struct WlanSSID {
    id: String,
    strength: u8,
}

#[derive(Deserialize, Debug)]
struct WlanUUID {
    con_uuid: String,
}

#[derive(Deserialize, Debug)]
struct APParam {
    bridge_addr: String,
    netmask: String,
    hw_mode: String,
    channel: String,
    ssid: String,
    wpa_passphrase: String,
}

#[derive(Deserialize, Debug)]
struct APDelParam {}

#[derive(Template)]
#[template(path = "../src/wlan/templates/wlan.html")]
struct WlanTemplate {
    interface: &'static str,
    ssid_list: Vec<WlanSSID>,
    info: WlanInfo,
    wlan_client: bool,
}

pub fn routes() -> Router {
    Router::new()
        .route("/wlan", get(wlan))
        .route("/wlan_scan", get(wlan_scan))
        .route("/wlan_setup", post(wlan_setup))
        .route("/wlan_save", post(wlan_save))
        .route("/wlan_delete", post(wlan_delete))
        .route("/wlan_connect", post(wlan_connect))
        .route("/wlan_disconnect", post(wlan_disconnect))
        .route("/ap_setup", post(ap_setup))
        .route("/ap_delete", post(ap_delete))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./wlan")
        }))
}

fn wlan_level(level: u8) -> Option<u8> {
    match level {
        0..=5 => Some(0),
        6..=30 => Some(1),
        31..=55 => Some(2),
        56..=79 => Some(3),
        80..=100 => Some(4),
        _ => None,
    }
}

async fn get_ssid_list(rescan: bool) -> Result<Vec<WlanSSID>> {
    let mut args = vec!["get_ssid_list.sh"];
    if !rescan {
        args.push("--no-rescan");
    }
    let output = common::exec_command(&args).await?;
    let stdout = String::from_utf8_lossy(&output.stdout);
    Ok(stdout
        .lines()
        .map(parse_nmcli_t_line)
        .filter_map(|a| {
            // field order as per get_ssid_list.sh:
            // SSID,SIGNAL,BSSID,CHAN,RATE,SECURITY
            let level = a.get(1)?.parse().ok()?;
            Some(WlanSSID {
                id: a.first()?.to_string(),
                strength: wlan_level(level)?,
            })
        })
        .collect())
}

async fn get_mac(interface: &Option<String>) -> Option<String> {
    let interface = interface.as_deref()?;
    let Ok(mut dev_info) = NMCon::from_device(interface).await else {
        return None;
    };
    dev_info.prop_take("GENERAL.HWADDR")
}

pub async fn get_wlan_info() -> Result<WlanInfo> {
    let mut wlan: Vec<NMCon> = join_all(
        nmconvec()
            .await?
            .iter()
            .filter(|c| c.ctype == "802-11-wireless")
            .map(|c| async move { NMCon::from_id(&c.uuid).await }),
    )
    .await
    .into_iter()
    .filter_map(|c| c.ok())
    .collect();

    // sta info
    let mut sta_info = None;
    // all wireless connections
    let mut wlan_saved = Vec::new();

    for con in wlan.iter_mut() {
        let Some(uuid) = con.prop_take("connection.uuid") else {
            continue;
        };
        let data = WlanSaved {
            uuid,
            // XXX check hidden AP
            ssid: con
                .prop_take("802-11-wireless.ssid")
                .unwrap_or_else(|| "--".to_string()),
            autoconnect: con
                .prop_take("connection.autoconnect")
                .unwrap_or_else(|| "--".to_string()),
        };
        if con.state == "activated" {
            let interface = con.prop_take("GENERAL.IP-IFACE");
            let mac_addr = get_mac(&interface).await;
            sta_info = Some(StaInfo {
                uuid: Some(data.uuid.clone()),
                ssid: Some(data.ssid.clone()),
                security: match con.prop_take("802-11-wireless-security.key-mgmt") {
                    Some(s) if s == "wpa-psk" => Some("WPA2".to_string()),
                    Some(s) if s == "sae" => Some("WPA3".to_string()),
                    other => other,
                },
                ip_addr: con.prop_take("IP4.ADDRESS[1]"),
                gw_addr: con.prop_take("IP4.GATEWAY"),
                interface,
                mac_addr,
            });
        }

        wlan_saved.push(data);
    }
    wlan_saved.sort_unstable_by(|a, b| a.ssid.cmp(&b.ssid));

    // hostapd config
    let hostapd = HostapdConfig::read().await?;
    // ap bridge address
    let mut bridge_addr = None;

    if let Some(bridge) = &hostapd.bridge {
        if let Ok(mut bridge_info) = NMCon::from_device(bridge).await {
            bridge_addr = bridge_info.prop_take("IP4.ADDRESS[1]");
        }
    }

    Ok(WlanInfo {
        sta: sta_info.unwrap_or_default(),
        wlan_saved,
        hostapd,
        bridge_addr,
    })
}

async fn wlan(Extension(title): Extension<Title>) -> PageResult {
    // XXX don't hardcode wlan_interface: get it from nmcli device list...
    let interface = common::wlan_interface().unwrap_or("--");

    let mut ssid_list = get_ssid_list(false).await?;
    // remove duplicates
    {
        let mut seen = HashSet::new();
        ssid_list.retain(|ssid| seen.insert(ssid.id.clone()));
    }

    let wlan_info = get_wlan_info().await?;
    let template = WlanTemplate {
        interface,
        ssid_list,
        info: wlan_info,
        wlan_client: Config::get().customize.get().wlan_client,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn wlan_scan(session: Session) -> ErrorStringResult {
    if !cfg!(debug_assertions) && !session.logged_in().await {
        return Ok((StatusCode::FORBIDDEN, "Not logged in").into_response());
    }
    let ssid_list = get_ssid_list(true).await?;
    let list = serde_json::to_string(&ssid_list).context("Could not format json.")?;
    let mut res = list.into_response();
    res.headers_mut()
        .insert("content-type", "application/json".parse().unwrap());
    Ok(res)
}

fn wlan_save_args(args: &mut Vec<String>, wlan_params: WlanParam) -> Result<()> {
    let interface = common::wlan_interface().context("Could not find wlan network interface.")?;
    args.push(wlan_params.essid);
    args.push(wlan_params.security);
    args.push(wlan_params.password);
    args.push(interface.to_string());
    args.push(wlan_params.method.clone());

    if wlan_params.method == "manual" {
        let netmask = if wlan_params.netmask.is_empty() {
            "255.255.255.255"
        } else {
            &wlan_params.netmask
        };
        let ip_addr = Ipv4Addr::from_str(&wlan_params.ip_addr)
            .with_context(|| format!("Could not parse addr: {}", wlan_params.ip_addr))?;
        let netmask = Ipv4Addr::from_str(netmask)
            .with_context(|| format!("Could not parse network: {netmask}"))?;
        let net = Ipv4Net::with_netmask(ip_addr, netmask)
            .with_context(|| format!("Could not define network with {ip_addr} / {netmask}"))?;

        args.push("-a".to_string());
        args.push(net.to_string());
        if !wlan_params.gateway_addr.is_empty() {
            args.push("-g".to_string());
            args.push(wlan_params.gateway_addr);
        }
        if !wlan_params.p_dns_addr.is_empty() {
            let mut dns = wlan_params.p_dns_addr;
            if !wlan_params.s_dns_addr.is_empty() {
                dns = format!("{},{}", dns, wlan_params.s_dns_addr);
            }
            args.push("-d".to_string());
            args.push(dns);
        }
    }
    Ok(())
}

async fn wlan_setup(Form(wlan_param): Form<WlanParam>) -> PageResult {
    if !Config::get().customize.get().wlan_client {
        return Err("Could not setup wlan client. This page has been disabled")?;
    }
    let mut args = vec!["wlan_setup.sh".to_string()];
    wlan_save_args(&mut args, wlan_param)?;
    args.push("--connect".to_string());

    common::exec_command(&args).await?;
    Ok(Redirect::to("/wlan").into_response())
}

async fn wlan_save(Form(wlan_param): Form<WlanParam>) -> PageResult {
    if !Config::get().customize.get().wlan_client {
        return Err("Could not save wlan client. This page has been disabled")?;
    }
    let mut args = vec!["wlan_setup.sh".to_string()];
    wlan_save_args(&mut args, wlan_param)?;

    common::exec_command(&args).await?;
    Ok(Redirect::to("/wlan").into_response())
}

async fn wlan_delete(Form(wlan): Form<WlanUUID>) -> PageResult {
    if !Config::get().customize.get().wlan_client {
        return Err("Could not delete wlan client. This page has been disabled")?;
    }
    nmcon_act("delete", &wlan.con_uuid).await?;

    Ok(Redirect::to("/wlan").into_response())
}

async fn wlan_connect(Form(wlan): Form<WlanUUID>) -> PageResult {
    if !Config::get().customize.get().wlan_client {
        return Err("Could not connect wlan client. This page has been disabled")?;
    }
    nmcon_act("up", &wlan.con_uuid).await?;

    Ok(Redirect::to("/wlan").into_response())
}

async fn wlan_disconnect(Form(wlan): Form<WlanUUID>) -> PageResult {
    if !Config::get().customize.get().wlan_client {
        return Err("Could not disconnect wlan client. This page has been disabled")?;
    }
    nmcon_act("down", &wlan.con_uuid).await?;

    Ok(Redirect::to("/wlan").into_response())
}

async fn ap_setup(Form(ap_param): Form<APParam>) -> PageResult {
    let netmask = if ap_param.netmask.is_empty() {
        "255.255.255.255"
    } else {
        &ap_param.netmask
    };
    let ip_addr = Ipv4Addr::from_str(&ap_param.bridge_addr)
        .with_context(|| format!("Could not parse addr: {}", &ap_param.bridge_addr))?;
    let netmask = Ipv4Addr::from_str(netmask)
        .with_context(|| format!("Could not parse network: {netmask}"))?;
    let net = Ipv4Net::with_netmask(ip_addr, netmask)
        .with_context(|| format!("Could not define network with {ip_addr} / {netmask}"))?;
    let interface = common::ap_interface().context("Could not find wlan network interface.")?;

    let args = &[
        "ap_setup.sh",
        common::AP_BRIDGE_NAME,
        &net.to_string(),
        &ap_param.hw_mode,
        &ap_param.channel,
        &ap_param.ssid,
        &ap_param.wpa_passphrase,
        interface,
    ];
    common::exec_command(args).await?;

    Ok(Redirect::to("/wlan").into_response())
}

async fn ap_delete(Form(_): Form<APDelParam>) -> PageResult {
    let interface = common::ap_interface().context("Could not find wlan network interface.")?;

    let args = &["ap_delete.sh", common::AP_BRIDGE_NAME, interface];
    common::exec_command(args).await?;

    Ok(Redirect::to("/wlan").into_response())
}
