// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Context, Result};
use axum::{
    body::Bytes,
    response::IntoResponse,
    routing::{delete, post},
    Router,
};
use axum_typed_multipart::{FieldData, TryFromMultipart, TypedMultipart};

use crate::common::{self, CheckAuthRestApi, RestApiPermissionVpnAdmin};
use crate::error::ErrorStringResult;
use crate::vpn::{vpn_act, VpnConfig};

#[derive(TryFromMultipart)]
struct VpnParam {
    setting_name: String,
    conf: FieldData<Bytes>,
    auth_type: String,
    username: Option<String>,
    password: Option<String>,
    cert: Option<FieldData<Bytes>>,
    key: Option<FieldData<Bytes>>,
    key_pass: Option<String>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/api/vpn/openvpn", post(rest_ovpn_setup))
        .route("/api/vpn/openvpn", delete(rest_ovpn_delete))
        .route("/api/vpn/up", post(rest_vpn_up))
        .route("/api/vpn/down", post(rest_vpn_down))
}

/// POST "/api/vpn/openvpn"
/// - Access: VpnAdmin
/// - Input: setting_name = <any string>
///   conf = <vpn config file path>
///   auth_type = "username" or "cert"
///   username = <username> (if auth_type = "username")
///   password = <password> (if auth_type = "username")
///   cert = <cert file path> (if auth_type = "cert")
///   key = <key file path> (if auth_type = "cert")
///   key_pass = <key password>
/// - Output: None
async fn rest_ovpn_setup(
    _auth: CheckAuthRestApi<RestApiPermissionVpnAdmin>,
    TypedMultipart(vpn_param): TypedMultipart<VpnParam>,
) -> ErrorStringResult {
    // Delete current settings before setup.
    let args = &["vpn_delete.sh"];
    common::exec_command(args).await?;

    let conf_contents = vpn_param.conf.contents;
    let conf_name = vpn_param
        .conf
        .metadata
        .file_name
        .context("Config file name is none")?;
    let vpn_config = VpnConfig::new()?;
    vpn_config.save_file(&conf_name, &conf_contents).await?;

    let mut args = vec![
        "vpn_setup.sh".to_string(),
        vpn_config.dir()?.to_string(),
        vpn_param.setting_name,
        vpn_param.auth_type.clone(),
        conf_name,
    ];

    match &*vpn_param.auth_type {
        "userpass" => {
            let username = vpn_param.username.context("username is required.")?;
            let password = vpn_param.password.context("password is required.")?;
            args.push("--user".to_string());
            args.push(username);
            args.push("--password".to_string());
            args.push(password);
        }
        "cert" => {
            if let Some(cert) = vpn_param.cert {
                let cert_contents = cert.contents;
                let cert_name = cert.metadata.file_name.context("Cert file name is none.")?;
                if !cert_contents.is_empty() && !cert_name.is_empty() {
                    vpn_config.save_file(&cert_name, &cert_contents).await?;
                    args.push("--cert".to_string());
                    args.push(cert_name);
                }
            }

            if let Some(key) = vpn_param.key {
                let key_contents = key.contents;
                let key_name = key.metadata.file_name.context("Key file name is none.")?;
                if !key_contents.is_empty() && !key_name.is_empty() {
                    vpn_config.save_file(&key_name, &key_contents).await?;
                    args.push("--key".to_string());
                    args.push(key_name);
                }
            }

            if let Some(key_pass) = vpn_param.key_pass {
                args.push("--askpass".to_string());
                args.push(key_pass);
            }
        }
        _ => Err(anyhow!("Bad auth_type {}.", vpn_param.auth_type))?,
    }

    args.push("--quiet".to_string());

    common::exec_command(&args).await?;
    Ok(().into_response())
}

/// DELETE "/api/vpn/openvpn"
/// - Access: VpnAdmin
/// - Input: None
/// - Output: None
async fn rest_ovpn_delete(_auth: CheckAuthRestApi<RestApiPermissionVpnAdmin>) -> ErrorStringResult {
    let args = &["vpn_delete.sh"];
    common::exec_command(args).await?;

    Ok(().into_response())
}

/// POST "/api/vpn/up"
/// - Access: VpnAdmin
/// - Input: None
/// - Output: None
async fn rest_vpn_up() -> ErrorStringResult {
    vpn_act(true).await?;

    Ok(().into_response())
}

/// POST "/api/vpn/down"
/// - Access: VpnAdmin
/// - Input: None
/// - Output: None
async fn rest_vpn_down() -> ErrorStringResult {
    vpn_act(false).await?;

    Ok(().into_response())
}
