// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use askama::Template;
use axum::{
    extract::Extension, middleware, response::IntoResponse, response::Redirect, routing::get,
    routing::post, Form, Router,
};

use crate::common::{
    check_auth, exec_command, get_title, usb_filter::AllowRulesClass, usb_filter::AllowRulesDevice,
    usb_filter::AllowRulesInfo, usb_filter::UsbDevice, usb_filter::UsbDeviceInfo, HtmlTemplate,
    Title,
};
use crate::error::PageResult;
use serde::Deserialize;

#[derive(Deserialize)]
struct ListDeviceParam {
    props: String,
    #[serde(default)]
    all_serial: bool,
}

#[derive(Deserialize)]
struct ListRuleParam {
    props: String,
}

#[derive(Deserialize)]
struct AllowClassParam {
    class: String,
}

#[derive(Template)]
#[template(path = "../src/usb_filter/templates/usb_filter.html")]
struct UsbFilterTemplate {
    status: String,
    usb_device_info: UsbDeviceInfo,
    allow_rules_info: AllowRulesInfo,
}

#[derive(Template)]
#[template(path = "../src/usb_filter/templates/usb_filter_device_detail_page.html")]
struct UsbFilterDeviceDetailTemplate {
    usb_device: UsbDevice,
}

#[derive(Template)]
#[template(path = "../src/usb_filter/templates/usb_filter_remove_rule_device_page.html")]
struct UsbFilterRemoveRuleDeviceTemplate {
    remove_rule_device: AllowRulesDevice,
}

#[derive(Template)]
#[template(path = "../src/usb_filter/templates/usb_filter_remove_rule_class_page.html")]
struct UsbFilterRemoveRuleClassTemplate {
    remove_rule_class: AllowRulesClass,
}

#[derive(Template)]
#[template(path = "../src/usb_filter/templates/usb_filter_allow_class_page.html")]
struct UsbFilterAllowClassTemplate {
    blocked_classes: Vec<String>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/usb_filter", get(usb_filter))
        .route("/usb_filter_enable", post(usb_filter_enable))
        .route("/usb_filter_disable", post(usb_filter_disable))
        .route("/usb_filter_allow_device", post(usb_filter_allow_device))
        .route(
            "/usb_filter_device_detail_page",
            get(usb_filter_device_detail_page),
        )
        .route(
            "/usb_filter_remove_rule_device_page",
            get(usb_filter_remove_rule_device_page),
        )
        .route(
            "/usb_filter_remove_rule_class_page",
            get(usb_filter_remove_rule_class_page),
        )
        .route("/usb_filter_remove_rule", post(usb_filter_remove_rule))
        .route(
            "/usb_filter_allow_class_page",
            get(usb_filter_allow_class_page),
        )
        .route("/usb_filter_allow_class", post(usb_filter_allow_class))
        .route_layer(middleware::from_fn(check_auth))
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./usb_filter")
        }))
}

async fn usb_filter(Extension(title): Extension<Title>) -> PageResult {
    let status = usb_filter_status().await?;
    let usb_device_info = UsbDeviceInfo::get().await?;
    let allow_rules_info = AllowRulesInfo::get().await?;
    let template = UsbFilterTemplate {
        status,
        usb_device_info,
        allow_rules_info,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn usb_filter_status() -> Result<String> {
    let args = &["usb_filter.sh", "status"];
    let output = exec_command(args).await?;
    let stdout = String::from_utf8_lossy(&output.stdout).to_string();
    Ok(stdout)
}

async fn usb_filter_enable() -> PageResult {
    let args = &["usb_filter.sh", "enable"];
    exec_command(args).await?;
    Ok(Redirect::to("/usb_filter").into_response())
}

async fn usb_filter_disable() -> PageResult {
    let args = &["usb_filter.sh", "disable"];
    exec_command(args).await?;
    Ok(Redirect::to("/usb_filter").into_response())
}

async fn usb_filter_allow_device(Form(list_device_param): Form<ListDeviceParam>) -> PageResult {
    let props = &list_device_param
        .props
        .split_ascii_whitespace()
        .collect::<Vec<&str>>();
    let vendor_id = props.get(3).context("invalid argument")?;
    let product_id = props.get(4).context("invalid argument")?;
    let model = props.get(5).context("invalid argument")?;
    let usb_interfaces = props.get(6).context("invalid argument")?;
    let mut serial = props.get(7).context("invalid argument")?;
    if list_device_param.all_serial {
        serial = &"*";
    }
    let args = &[
        "usb_filter.sh",
        "allow_device",
        vendor_id,
        product_id,
        model,
        usb_interfaces,
        serial,
    ];
    exec_command(args).await?;
    Ok(Redirect::to("/usb_filter").into_response())
}

async fn usb_filter_device_detail_page(
    Form(list_device_param): Form<ListDeviceParam>,
) -> PageResult {
    let props = &list_device_param
        .props
        .split_ascii_whitespace()
        .collect::<Vec<&str>>();
    let default_str = &"--";
    let template = UsbFilterDeviceDetailTemplate {
        usb_device: UsbDevice {
            list_id: props.first().unwrap_or(default_str).to_string(),
            permission: props.get(1).unwrap_or(default_str).to_string(),
            bus_device_num: props.get(2).unwrap_or(default_str).to_string(),
            vendor_id: props.get(3).unwrap_or(default_str).to_string(),
            product_id: props.get(4).unwrap_or(default_str).to_string(),
            model: props.get(5).unwrap_or(default_str).to_string(),
            usb_interfaces: props.get(6).unwrap_or(default_str).to_string(),
            serial: props.get(7).unwrap_or(default_str).to_string(),
            dir_pass: props.get(8).unwrap_or(default_str).to_string(),
        },
    };
    Ok(HtmlTemplate::new("詳細表示", template).into_response())
}

async fn usb_filter_remove_rule_device_page(
    Form(list_rule_param): Form<ListRuleParam>,
) -> PageResult {
    let list_id = &list_rule_param
        .props
        .split_ascii_whitespace()
        .next()
        .context("invalid argument")?;
    let args = &["usb_filter.sh", "get_rule", list_id];
    let output = &exec_command(args).await?.stdout;
    let rule = &String::from_utf8_lossy(output);
    let props = rule
        .split_whitespace()
        .map(|x| x.replace("\"", ""))
        .collect::<Vec<String>>();
    let default_str = &"--".to_string();
    let remove_rule_device = AllowRulesDevice {
        list_id: props.first().unwrap_or(default_str).to_string(),
        vendor_id: props.get(2).unwrap_or(default_str).to_string(),
        product_id: props.get(3).unwrap_or(default_str).to_string(),
        model: props.get(4).unwrap_or(default_str).to_string(),
        usb_interfaces: props.get(5).unwrap_or(default_str).to_string(),
        serial: props.get(6).unwrap_or(default_str).to_string(),
    };
    let template = UsbFilterRemoveRuleDeviceTemplate { remove_rule_device };
    Ok(HtmlTemplate::new("削除", template).into_response())
}

async fn usb_filter_remove_rule_class_page(
    Form(list_rule_param): Form<ListRuleParam>,
) -> PageResult {
    let list_id = &list_rule_param
        .props
        .split_ascii_whitespace()
        .next()
        .context("invalid argument")?;
    let args = &["usb_filter.sh", "get_rule", list_id];
    let output = &exec_command(args).await?.stdout;
    let rule = &String::from_utf8_lossy(output);
    let props = rule.split_whitespace().collect::<Vec<&str>>();
    let default_str = &"--";
    let remove_rule_class = AllowRulesClass {
        list_id: props.first().unwrap_or(default_str).to_string(),
        class: props.get(2).unwrap_or(default_str).to_string(),
    };
    let template = UsbFilterRemoveRuleClassTemplate { remove_rule_class };
    Ok(HtmlTemplate::new("削除", template).into_response())
}

async fn usb_filter_remove_rule(Form(list_rule_param): Form<ListRuleParam>) -> PageResult {
    let list_id = &list_rule_param
        .props
        .split_ascii_whitespace()
        .next()
        .context("invalid argument")?;
    let args = &["usb_filter.sh", "remove_rule", list_id];
    exec_command(args).await?;
    Ok(Redirect::to("/usb_filter").into_response())
}

async fn usb_filter_allow_class_page() -> PageResult {
    let args = &["usb_filter.sh", "list_block_classes"];
    let output = &exec_command(args).await?.stdout;
    let rule = &String::from_utf8_lossy(output);
    let blocked_classes = rule.split_whitespace().map(|s| s.to_string()).collect();
    let template = UsbFilterAllowClassTemplate { blocked_classes };
    Ok(HtmlTemplate::new("USBデバイスクラスの追加", template).into_response())
}

async fn usb_filter_allow_class(Form(allow_class_param): Form<AllowClassParam>) -> PageResult {
    let args = &["usb_filter.sh", "allow_class", &allow_class_param.class];
    exec_command(args).await?;
    Ok(Redirect::to("/usb_filter").into_response())
}
