// SPDX-License-Identifier: MIT

use anyhow::Context;
use axum::{
    response::IntoResponse,
    routing::{get, post},
    Json, Router,
};
use serde_json::json;
use std::time::{SystemTime, UNIX_EPOCH};

use crate::common::{
    json_response, CheckAuthRestApi, JsonOrForm, RestApiPermissionTimeAdmin,
    RestApiPermissionTimeView,
};
use crate::error::ErrorStringResult;
use crate::time_settings::{ntp, time_sync, timezone};

pub fn routes() -> Router {
    Router::new()
        .route("/api/time/ntp_config", get(rest_get_ntp_config))
        .route("/api/time/ntp_info", get(rest_get_ntp_info))
        .route("/api/time/ntp_config", post(rest_post_ntp_config))
        .route("/api/time/timezone", get(rest_get_timezone))
        .route("/api/time/timezone", post(rest_post_timezone))
        .route("/api/time/set", post(rest_post_time_set))
}

/// GET "/api/time/ntp_config"
/// - Access: time view
/// - Input: None
/// - Output: json object with:
///   - servers: list of configured servers (lines of chronyd config)
///   - initstepslew: bool
async fn rest_get_ntp_config(
    _auth: CheckAuthRestApi<RestApiPermissionTimeView>,
) -> ErrorStringResult {
    Ok(Json(ntp::get_config().await?).into_response())
}

/// GET "/api/time/ntp_info"
/// - Access: time view
/// - Input: None
/// - Output: json object with:
///   - time_now: unix epoch of device
///   - ntp_server_ip: ip of server currently synchronized if synchronized
///   - ntp_server_offset: offset with server representing time behind the server if synchronized
async fn rest_get_ntp_info(
    _auth: CheckAuthRestApi<RestApiPermissionTimeView>,
) -> ErrorStringResult {
    let info = ntp::get_info().await?;
    json_response(&json!({
        "time_now": SystemTime::now().duration_since(UNIX_EPOCH).context("Bad time")?.as_secs(),
        "ntp_server_ip": info.as_ref().map(|i| &i.server_ip),
        "ntp_server_offset": info.as_ref().map(|i| &i.offset)
    }))
}

/// POST "/api/time/ntp_config"
/// - Access: time admin
/// - Input: optional parameters (at least one must be set):
///   - servers: list of configured servers (lines of chronyd config)
///   - initstepslew: bool
/// - Output: modified settings (same as get)
async fn rest_post_ntp_config(
    _auth: CheckAuthRestApi<RestApiPermissionTimeAdmin>,
    JsonOrForm(settings): JsonOrForm<ntp::NtpSettingsParam>,
) -> ErrorStringResult {
    ntp::set_config(settings).await?;
    Ok(Json(ntp::get_config().await?).into_response())
}

/// GET "/api/time/timezone"
/// - Access: time view
/// - Input: None
/// - Output: json object with:
///   - timezone: current timezone
async fn rest_get_timezone(
    _auth: CheckAuthRestApi<RestApiPermissionTimeView>,
) -> ErrorStringResult {
    json_response(&json!({
        "timezone": timezone::get_timezone().await?,
    }))
}

/// POST "/api/time/timezone"
/// - Access: time admin
/// - Input: 'timezone' parameter
/// - Output: None
async fn rest_post_timezone(
    _auth: CheckAuthRestApi<RestApiPermissionTimeAdmin>,
    JsonOrForm(param): JsonOrForm<timezone::TimezoneParam>,
) -> ErrorStringResult {
    timezone::set_timezone(param).await?;
    Ok(().into_response())
}

/// POST "/api/time/set"
/// - Access: time admin
/// - Input: 'timestamp' parameter, unix epoch in second
/// - Output: None
async fn rest_post_time_set(
    _auth: CheckAuthRestApi<RestApiPermissionTimeAdmin>,
    JsonOrForm(param): JsonOrForm<time_sync::TimeSyncParam>,
) -> ErrorStringResult {
    time_sync::time_sync(param).await?;
    Ok(().into_response())
}
