// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    extract::Extension,
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use axum_extra::extract::Form;

use crate::common::{check_auth, get_title, HtmlTemplate, Title};
use crate::error::PageResult;

mod ntp;
mod time_sync;
mod timezone;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

pub fn routes() -> Router {
    Router::new()
        .route("/time", get(time))
        .route("/time_ntp_setup", post(time_ntp_setup))
        .route("/time_ntp_reset", post(time_ntp_reset))
        .route("/time_zone_setup", post(timezone_setup))
        .route("/time_sync", post(time_sync_timestamp))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./time")
        }))
}

struct NtpDisplayInfo<'a> {
    ip: &'a str,
    offset: &'a str,
    ahead: bool,
}

#[derive(Template)]
#[template(path = "../src/time_settings/templates/time.html")]
struct TimeTemplate<'a> {
    now: String,
    ntp_info: Option<NtpDisplayInfo<'a>>,
    ntp_servers: Vec<String>,
    initstepslew: &'a str,
    timezones: Vec<String>,
    current_timezone: String,
}

async fn time(Extension(title): Extension<Title>) -> PageResult {
    let (ntp_config, ntp_info, timezones, current_timezone) = futures::try_join!(
        ntp::get_config(),
        ntp::get_info(),
        timezone::list_timezones(),
        timezone::get_timezone()
    )?;
    let ntp_info = match &ntp_info {
        None => None,
        Some(info) => {
            let ahead = info.offset.starts_with('-');
            Some(NtpDisplayInfo {
                ip: &info.server_ip,
                offset: if ahead {
                    // does not panic because ahead ensured we had a single ascii char in position 0
                    &info.offset[1..]
                } else {
                    &info.offset
                },
                ahead,
            })
        }
    };
    let template = TimeTemplate {
        now: chrono::Local::now()
            .format("%Y-%m-%d %H:%M:%S %Z")
            .to_string(),
        ntp_info,
        ntp_servers: ntp_config.servers,
        initstepslew: ntp_config.initstepslew.as_deref().unwrap_or(""),
        timezones,
        current_timezone,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn time_ntp_setup(Form(ntp_param): Form<ntp::NtpSettingsParam>) -> PageResult {
    ntp::set_config(ntp_param).await?;
    Ok(Redirect::to("/time").into_response())
}

async fn time_ntp_reset() -> PageResult {
    ntp::set_config(ntp::NtpSettingsParam {
        servers: vec!["default".to_string()],
        initstepslew: vec!["default".to_string()],
    })
    .await?;
    Ok(Redirect::to("/time").into_response())
}

async fn timezone_setup(Form(timezone_param): Form<timezone::TimezoneParam>) -> PageResult {
    timezone::set_timezone(timezone_param).await?;
    Ok(Redirect::to("/time").into_response())
}

async fn time_sync_timestamp(Form(param): Form<time_sync::TimeSyncParam>) -> PageResult {
    time_sync::time_sync(param).await?;
    Ok(Redirect::to("/time").into_response())
}
