// SPDX-License-Identifier: MIT

use anyhow::Context;
use axum::{
    extract::{DefaultBodyLimit, Multipart},
    response::IntoResponse,
    routing::{get, post},
    Router,
};
use serde::{Deserialize, Serialize};
use std::collections::HashMap;

use crate::common::{
    json_response, json_stream_command, multipart_to_input_chan, CheckAuthRestApi, CommandOpts,
    JsonOrForm, RestApiPermissionSwuInstall, RestApiPermissionSwuView,
};
use crate::error::ErrorStringResult;
use crate::swu::{get_last_update, get_swu_versions};

pub fn routes() -> Router {
    Router::new()
        .layer(DefaultBodyLimit::max(2048 * 1024))
        .route("/api/swu/install/upload", post(rest_swu_install_upload))
        .layer(DefaultBodyLimit::disable())
        .route("/api/swu/install/url", post(rest_swu_install_url))
        .route("/api/swu/versions", get(rest_swu_versions))
        .route("/api/swu/status", get(rest_swu_status))
}

/// POST "/api/swu/install/upload"
/// - Access: SwuInstall
/// - Input: multipart/form-data with swu content
/// - Output: json stream of object with 'line' items, followed by
///   either exit_code or exit_signal
async fn rest_swu_install_upload(
    _auth: CheckAuthRestApi<RestApiPermissionSwuInstall>,
    multipart: Multipart,
) -> impl IntoResponse {
    let input = multipart_to_input_chan(multipart).await;

    json_stream_command(
        vec!["swupdate.sh".to_string(), "stdin".to_string()],
        &CommandOpts {
            stream_ignore_output_errors: true,
            ..Default::default()
        },
        Some(input),
    )
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct SwuUrl {
    url: String,
}

/// POST "/api/swu/install/url"
/// - Access: SwuInstall
/// - Input: url={url}
/// - Output: json stream of object with 'line' items, followed by
///   either exit_code or exit_signal
async fn rest_swu_install_url(
    _auth: CheckAuthRestApi<RestApiPermissionSwuInstall>,
    JsonOrForm(param): JsonOrForm<SwuUrl>,
) -> impl IntoResponse {
    json_stream_command(
        vec!["swupdate.sh".to_string(), "url".to_string(), param.url],
        &CommandOpts {
            stream_ignore_output_errors: true,
            ..Default::default()
        },
        None,
    )
}

/// GET "/api/swu/versions"
/// - Access: SwuView
/// - Input: None
/// - Output: json object mapping each component to their versions
async fn rest_swu_versions(_auth: CheckAuthRestApi<RestApiPermissionSwuView>) -> ErrorStringResult {
    let swu_versions: HashMap<String, String> = get_swu_versions()
        .await?
        .into_iter()
        .map(|v| (v.component, v.version))
        .collect();
    json_response(&swu_versions)
}

#[derive(Serialize)]
struct RestSwuStatus {
    rollback_ok: bool,
    last_update_timestamp: i64,
    last_update_versions: HashMap<String, (Option<String>, Option<String>)>,
}

/// GET "/api/swu/status"
/// - Access: SwuView
/// - Input: None
/// - Output: json object with:
///   - rollback_ok: bool
///   - last_update_timestamp: UTC unix epoch (int)
///   - last_update_versions: component -> [old, new] object -- old can be 'null'
async fn rest_swu_status(_auth: CheckAuthRestApi<RestApiPermissionSwuView>) -> ErrorStringResult {
    let swu_versions = get_swu_versions().await?;
    let infos = get_last_update(&swu_versions)
        .await
        .context("No update found")?;
    json_response(&RestSwuStatus {
        rollback_ok: infos.rootfs == infos.updated_rootfs,
        last_update_timestamp: infos.updated_date.timestamp(),
        last_update_versions: infos
            .updated_versions
            .into_iter()
            .map(|v| (v.component, (v.old_version, v.new_version)))
            .collect(),
    })
}
