// SPDX-License-Identifier: MIT

use anyhow::{Error, Result};
use askama::Template;
use axum::{extract::Extension, middleware, response::IntoResponse, routing::get, Router};
use std::collections::HashMap;

use crate::common::{
    self, check_auth, container::ContainerInfo, dhcp_list, filters, get_title, Config, DhcpInfo,
    HtmlTemplate, IptablesInfo, NatInfo, PortForwardingInfo, Title,
};
use crate::error::PageResult;
use crate::vpn;
use crate::wlan::{get_wlan_info, WlanInfo};

#[derive(Template)]
#[template(path = "../src/status/templates/status.html")]
struct StatusTemplate {
    wwan_info: HashMap<&'static str, String>,
    wwan_addr: HashMap<&'static str, String>,
    wlan_info: WlanInfo,
    wlan_client: bool,
    lan_info: Vec<HashMap<&'static str, String>>,
    dhcp_info: Vec<DhcpInfo>,
    nat_info: Vec<NatInfo>,
    nat_enabled: bool,
    pf_info: Vec<PortForwardingInfo>,
    container_info: ContainerInfo,
    vpn_info: HashMap<&'static str, String>,
    vpn_addr: HashMap<&'static str, String>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/status", get(status))
        .route_layer(middleware::from_fn(check_auth))
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./status")
        }))
}

async fn status(Extension(title): Extension<Title>) -> PageResult {
    let (wwan_addr, wwan_info) = wwan_info().await?;
    let wlan_info = get_wlan_info().await?;
    let lan_info = lan_info().await?;
    let dhcp_info = dhcp_list().await?;
    let iptables_info = IptablesInfo::get().await?;
    let container_info = ContainerInfo::get().await?;
    let (vpn_addr, vpn_info) = vpn_info().await?;
    let template = StatusTemplate {
        wwan_addr,
        wwan_info,
        wlan_info,
        wlan_client: Config::get().customize.get().wlan_client,
        lan_info,
        dhcp_info,
        nat_info: iptables_info.nat,
        nat_enabled: Config::get().customize.get().nat_enabled,
        pf_info: iptables_info.port_forwarding,
        container_info,
        vpn_addr,
        vpn_info,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

pub async fn wwan_info(
) -> Result<(HashMap<&'static str, String>, HashMap<&'static str, String>), Error> {
    let wwan_info = match common::wwan_info(common::WWAN_CON_NAME).await {
        Ok(wwan) => wwan,
        Err(e) => return Err(Error::msg(e)),
    };

    let interface = common::wwan_interface().unwrap_or("--");
    let wwan_addr = match common::current_address("ttyCommModem", interface).await {
        Ok(addr) => addr,
        Err(e) => return Err(Error::msg(e)),
    };
    Ok((wwan_addr, wwan_info))
}

async fn lan_info() -> Result<Vec<HashMap<&'static str, String>>, Error> {
    let interface = "eth0";
    let lan_addr = match common::current_address(interface, interface).await {
        Ok(addr) => addr,
        Err(e) => return Err(Error::msg(e)),
    };
    let mut lan_addrs: Vec<HashMap<&'static str, String>> = Vec::new();
    lan_addrs.push(lan_addr);

    let exists_eth1 = common::exists_eth1();
    if exists_eth1 {
        let interface = "eth1";
        let lan_addr = match common::current_address(interface, interface).await {
            Ok(addr) => addr,
            Err(e) => return Err(Error::msg(e)),
        };
        lan_addrs.push(lan_addr);
    }
    Ok(lan_addrs)
}

pub async fn vpn_info(
) -> Result<(HashMap<&'static str, String>, HashMap<&'static str, String>), Error> {
    let args = &["vpn_info.sh"];
    let output = common::exec_command(args).await?;
    let setting_name = String::from_utf8_lossy(&output.stdout).trim().to_string();
    let mut vpn_info = HashMap::new();
    let connection_type = vpn::vpn_connection_type().await;
    vpn_info.insert("setting_name", setting_name);

    let mut interface = "tun0";
    if connection_type == vpn::VpnType::Wireguard {
        interface = "wg";
    }
    let vpn_addr = match common::current_address(interface, interface).await {
        Ok(addr) => addr,
        Err(e) => return Err(Error::msg(e)),
    };
    Ok((vpn_addr, vpn_info))
}
