// SPDX-License-Identifier: MIT

use axum::{response::IntoResponse, routing::post, Router};
use serde::Deserialize;

use crate::common::{
    exec_command, CheckAuthRestApi, JsonOrForm, RestApiPermissionAbosWebRestart,
    RestApiPermissionAdmin,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/abosweb/restart", post(restapi_abosweb_restart))
        .route("/api/sshd", post(restapi_sshd_post))
}

/// POST "/api/abosweb/restart"
/// - Access: AbosWebRestart
/// - Input: None
/// - Output: connection reset. (abos-web is stopped before returning a response)
async fn restapi_abosweb_restart(
    _auth: CheckAuthRestApi<RestApiPermissionAbosWebRestart>,
) -> ErrorStringResult {
    exec_command(&["abosweb_restart.sh"]).await?;
    Ok(().into_response())
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct SshdParam {
    sshd_enable: Option<bool>,
    rootpw_enable: Option<bool>,
    pubkey: Option<String>,
}

/// POST "/api/abosweb/sshd"
/// - Access: admin
/// - Input: optional enable, optional rootpw_enable, optional pubkey
/// - Output: None on success
///
/// Note there is no GET to check status because if enabled one can
/// just as easily check if port 22 is open
async fn restapi_sshd_post(
    _auth: CheckAuthRestApi<RestApiPermissionAdmin>,
    JsonOrForm(param): JsonOrForm<SshdParam>,
) -> ErrorStringResult {
    super::sshd::sshd_setup(param.sshd_enable, param.rootpw_enable, param.pubkey).await?;
    Ok(().into_response())
}
