// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Context, Result};
use askama::Template;
use axum::{
    body::Bytes,
    extract::ws::{self, WebSocket, WebSocketUpgrade},
    extract::Extension,
    http::StatusCode,
    middleware,
    response::{IntoResponse, Redirect, Response},
    routing::{get, post},
    Router,
};
use chrono::Local;
use futures::{sink::SinkExt, stream::StreamExt};
use hyper::header;
use std::collections::HashMap;
use std::path::Path;
use tokio::fs;
use tokio::io::AsyncWriteExt;
use tower_sessions::Session;
// need axum_extra's form to handle permissions list
use axum_extra::extract::Form;
use axum_typed_multipart::{FieldData, TryFromMultipart, TypedMultipart};
use serde::Deserialize;
use tempfile::TempDir;

use crate::common::{
    self, check_auth, check_password, exec_command_stdin, get_title, stream_command, CommandOpts,
    Config, Customize, HtmlTemplate, LoggedIn, Title,
};
use crate::error::{ErrorStringResult, PageResult};

#[cfg(feature = "restapi")]
use crate::common::{restapi, RestApiPermission, RestApiToken};

#[cfg(feature = "restapi")]
mod restapi_rest_tokens;
#[cfg(not(feature = "restapi"))]
mod restapi_rest_tokens {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[cfg(feature = "restapi")]
mod restapi_abosweb;
#[cfg(not(feature = "restapi"))]
mod restapi_abosweb {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[cfg(feature = "restapi")]
mod restapi_reset_default;
#[cfg(not(feature = "restapi"))]
mod restapi_reset_default {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

mod sshd;

const STATIC_CUSTOMIZE_DISABLE: &str = "/etc/atmark/abos_web/customize_disable";
const MAX_UPLOAD_SIZE: usize = 3 * 1024 * 1024;

#[cfg(feature = "restapi")]
#[derive(Template)]
#[template(path = "../src/settings/templates/settings_restapi.html")]
struct SettingsRestapiTemplate {
    tokens: Vec<RestApiToken>,
    tokens_enabled: bool,
}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings.html")]
struct SettingsTemplate {
    #[cfg(feature = "restapi")]
    restapi: SettingsRestapiTemplate,
    #[cfg(not(feature = "restapi"))]
    restapi: &'static str,
    log: SettingsLogTemplate,
    reset_default: SettingsResetDefaultTemplate,
    reset_default_log: SettingsResetDefaultLog,
    customize: SettingsCustomizeTemplate,
    customize_about_favicon: SettingsCustomizeAboutFavicon,
    sshd: sshd::SettingsSshdTemplate,
}

#[cfg(feature = "restapi")]
#[derive(Template)]
#[template(path = "../src/settings/templates/settings_restapi_setup.html")]
struct SettingsRestapiSetupTemplate {
    token_id: Option<String>,
    permissions: Vec<RestApiPermission>,
}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_log.html")]
struct SettingsLogTemplate {}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_reset_default.html")]
struct SettingsResetDefaultTemplate {}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_reset_default_log.html")]
struct SettingsResetDefaultLog {}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_customize.html")]
struct SettingsCustomizeTemplate {
    customize_disable: bool,
    header_title: String,
    model_number: String,
    color: String,
}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_customize_about_favicon.html")]
struct SettingsCustomizeAboutFavicon {
    customize_disable: bool,
}

#[derive(Template)]
#[template(path = "../src/settings/templates/settings_customize_menu.html")]
struct SettingsCustomizeMenuTemplate {
    display_menu: Vec<DisplayMenu>,
}

struct DisplayMenu {
    pub text: String,
    pub default_text: String,
    pub description: String,
}

pub fn routes() -> Router {
    let routes = Router::new()
        .route("/settings", get(settings))
        .route("/settings_password", post(settings_password))
        .route("/settings_password_reset", post(settings_password_reset))
        .route("/settings_customize_setup", post(settings_customize_setup))
        .route("/settings_customize_reset", post(settings_customize_reset))
        .route(
            "/settings_customize_disable",
            post(settings_customize_disable),
        )
        .route("/settings_customize_menu", get(settings_customize_menu))
        .route(
            "/settings_customize_menu_setup",
            post(settings_customize_menu_setup),
        )
        .route("/settings_reset_default_ws", get(settings_reset_default_ws))
        .route("/settings_get_log", get(settings_get_log))
        .route("/settings_download_log", get(settings_download_log))
        .route("/settings_sshd_setup", post(sshd::settings_sshd_setup));

    #[cfg(feature = "restapi")]
    let routes = routes
        .route("/settings_rest_token_delete", post(settings_token_delete))
        .route("/settings_rest_token_setup", get(settings_token_setup))
        .route("/settings_rest_token_save", post(settings_token_save));

    routes
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi_rest_tokens::routes())
        .merge(restapi_abosweb::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./settings")
        }))
        .merge(restapi_reset_default::routes())
}

async fn settings(Extension(title): Extension<Title>) -> PageResult {
    #[cfg(feature = "restapi")]
    let mut tokens = restapi::list_tokens().await?;
    #[cfg(feature = "restapi")]
    tokens.sort_unstable_by(|a, b| a.token_id.cmp(&b.token_id));
    let customize = &Config::get().customize.get();
    let template = SettingsTemplate {
        #[cfg(feature = "restapi")]
        restapi: SettingsRestapiTemplate {
            tokens,
            tokens_enabled: customize.tokens_enabled,
        },
        #[cfg(not(feature = "restapi"))]
        restapi: "",
        log: SettingsLogTemplate {},
        reset_default: SettingsResetDefaultTemplate {},
        reset_default_log: SettingsResetDefaultLog {},
        customize: SettingsCustomizeTemplate {
            customize_disable: Path::new(STATIC_CUSTOMIZE_DISABLE).exists(),
            header_title: customize.header_title.clone(),
            model_number: customize.model_number.clone().unwrap_or_default(),
            color: customize.color.clone(),
        },
        customize_about_favicon: SettingsCustomizeAboutFavicon {
            customize_disable: Path::new(STATIC_CUSTOMIZE_DISABLE).exists(),
        },
        sshd: sshd::sshd_settings().await,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

#[derive(Deserialize)]
struct ChangePasswordParam {
    current_password: String,
    password: String,
    password_confirm: String,
}

#[derive(Deserialize)]
struct ResetPasswordParam {
    current_password: String,
}

async fn settings_password(Form(param): Form<ChangePasswordParam>) -> PageResult {
    check_password(&param.current_password).await?;
    if param.password.chars().count() < 8 {
        Err("新しいパスワードが 8 文字未満です。")?;
    }
    if param.password != param.password_confirm {
        Err("新しいパスワードが一致しません。")?;
    }
    exec_command_stdin(&["register_password.sh"], &param.password).await?;
    Ok(Redirect::to("/settings").into_response())
}

async fn settings_password_reset(Form(param): Form<ResetPasswordParam>) -> PageResult {
    check_password(&param.current_password).await?;
    exec_command_stdin(&["register_password.sh", "-e"], "!").await?;
    Ok(Redirect::to("/settings").into_response())
}

#[cfg(feature = "restapi")]
#[derive(Deserialize)]
struct TokenIdParam {
    token_id: String,
}

#[cfg(feature = "restapi")]
async fn settings_token_delete(Form(param): Form<TokenIdParam>) -> PageResult {
    restapi::delete_token(&param.token_id).await?;
    Ok(Redirect::to("/settings").into_response())
}

#[cfg(feature = "restapi")]
#[derive(Deserialize)]
struct TokenIdOrNewParam {
    token_id: Option<String>,
    new_token: Option<bool>,
}

#[cfg(feature = "restapi")]
async fn settings_token_setup(Form(param): Form<TokenIdOrNewParam>) -> PageResult {
    let (token_id, permissions) = if param.new_token.unwrap_or(false) {
        (None, vec![RestApiPermission::Admin])
    } else {
        let token = restapi::get_token(&param.token_id.context("No token given")?).await?;
        (Some(token.token_id), token.permissions)
    };
    let template = SettingsRestapiSetupTemplate {
        token_id,
        permissions,
    };
    Ok(HtmlTemplate::new("Rest API トークン編集", template).into_response())
}

#[cfg(feature = "restapi")]
#[derive(Deserialize)]
struct TokenParam {
    token_id: Option<String>,
    permissions: Vec<RestApiPermission>,
}

#[cfg(feature = "restapi")]
async fn settings_token_save(Form(param): Form<TokenParam>) -> PageResult {
    match param.token_id {
        None => {
            restapi::create_token(param.permissions).await?;
        }
        Some(token_id) => {
            let mut token = restapi::get_token(&token_id).await.unwrap_or(RestApiToken {
                token_id,
                permissions: vec![],
            });
            token.permissions = param.permissions;
            restapi::write_token(&token).await?;
        }
    }
    Ok(Redirect::to("/settings").into_response())
}

#[derive(TryFromMultipart)]
struct CustomizeParam {
    logo_login_image: FieldData<Bytes>,
    logo_header_image: FieldData<Bytes>,
    header_title: Option<String>,
    model_number: Option<String>,
    favicon_zip: FieldData<Bytes>,
    color_pattern: String,
}

async fn settings_customize_setup(
    TypedMultipart(customize_param): TypedMultipart<CustomizeParam>,
) -> PageResult {
    if Path::new(STATIC_CUSTOMIZE_DISABLE).exists() {
        return Ok(Redirect::to("/top").into_response());
    }
    let mut customize = Config::get().customize.get_clone();
    customize.header_title = customize_param.header_title.unwrap_or_default();
    customize.model_number = customize_param.model_number;
    customize.color.clone_from(&customize_param.color_pattern);
    Config::get().customize.save(customize).await?;

    let tmp_dir = TempDir::new().context("Could not create temporary directory")?;
    let dir_path = tmp_dir.path();
    let mut args = vec!["customize_setup.sh".to_string()];

    args.push("--tmp_dir".to_string());
    let path_str = dir_path
        .to_str()
        .with_context(|| anyhow!("Bad path {:?}.", dir_path))?;
    args.push(path_str.to_string());

    if !customize_param.logo_login_image.contents.is_empty() {
        if customize_param.logo_login_image.contents.len() > MAX_UPLOAD_SIZE {
            return Err(anyhow!(
                "Could not upload logo login image. File size is limited to {} B.",
                MAX_UPLOAD_SIZE
            )
            .into());
        }

        let path = dir_path.join("logo_login.png");
        if let Err(e) = save_customize_file(&path, &customize_param.logo_login_image.contents).await
        {
            return Err(anyhow!("Could not save file: {}.", e).into());
        }
        args.push("--logo_login".to_string());
        let path_str = path
            .to_str()
            .with_context(|| anyhow!("Bad path {:?}.", path))?;
        args.push(path_str.to_string());
    }
    if !customize_param.logo_header_image.contents.is_empty() {
        if customize_param.logo_header_image.contents.len() > MAX_UPLOAD_SIZE {
            return Err(anyhow!(
                "Could not upload logo heder image. File size is limited to {} B.",
                MAX_UPLOAD_SIZE
            )
            .into());
        }

        let path = dir_path.join("logo_header.png");
        if let Err(e) =
            save_customize_file(&path, &customize_param.logo_header_image.contents).await
        {
            return Err(anyhow!("Could not save file: {}.", e).into());
        }
        args.push("--logo_header".to_string());
        let path_str = path
            .to_str()
            .with_context(|| anyhow!("Bad path {:?}.", path))?;
        args.push(path_str.to_string());
    }
    if !customize_param.favicon_zip.contents.is_empty() {
        if customize_param.favicon_zip.contents.len() > MAX_UPLOAD_SIZE {
            return Err(anyhow!(
                "Could not upload favicon zip. File size is limited to {} B.",
                MAX_UPLOAD_SIZE
            )
            .into());
        }

        let path = dir_path.join("favicon.zip");
        if let Err(e) = save_customize_file(&path, &customize_param.favicon_zip.contents).await {
            return Err(anyhow!("Could not save file: {}.", e).into());
        }
        args.push("--favicon".to_string());
        let path_str = path
            .to_str()
            .with_context(|| anyhow!("Bad path {:?}.", path))?;
        args.push(path_str.to_string());
    }
    args.push("--color".to_string());
    args.push(customize_param.color_pattern);
    common::exec_command(&args).await?;

    Ok(Redirect::to("/logout").into_response())
}

async fn settings_customize_reset() -> PageResult {
    if Path::new(STATIC_CUSTOMIZE_DISABLE).exists() {
        return Ok(Redirect::to("/top").into_response());
    }
    let args = &["customize_reset.sh"];
    common::exec_command(args).await?;
    Config::get().customize.reset();
    Ok(Redirect::to("/logout").into_response())
}

async fn settings_customize_disable() -> PageResult {
    let args = &["customize_disable.sh"];
    common::exec_command(args).await?;
    Ok(Redirect::to("/logout").into_response())
}

async fn settings_customize_menu() -> PageResult {
    if Path::new(STATIC_CUSTOMIZE_DISABLE).exists() {
        return Ok(Redirect::to("/top").into_response());
    }
    let customize = Config::get().customize.get();
    let side_menu_customize = &customize.side_menu;
    let side_menu_default = Customize::default().side_menu;
    let mut display_menu = Vec::<DisplayMenu>::new();
    for menu_d in side_menu_default {
        if let Some(menu_c) = side_menu_customize
            .iter()
            .find(|&m| m.href_path == menu_d.href_path)
        {
            display_menu.push(DisplayMenu {
                text: menu_c.text.clone(),
                default_text: menu_d.text,
                description: menu_c.description.clone(),
            });
        } else {
            display_menu.push(DisplayMenu {
                text: "".to_string(),
                default_text: menu_d.text,
                description: "".to_string(),
            });
        }
    }

    let template = SettingsCustomizeMenuTemplate { display_menu };
    Ok(HtmlTemplate::new("メニュー項目の変更", template).into_response())
}

async fn settings_customize_menu_setup(Form(props): Form<HashMap<String, String>>) -> PageResult {
    if Path::new(STATIC_CUSTOMIZE_DISABLE).exists() {
        return Ok(Redirect::to("/top").into_response());
    }

    let mut customize = Config::get().customize.get_clone();
    let mut side_menu = Customize::default().side_menu;
    for (k, v) in props {
        match k.chars().next() {
            Some('t') => {
                let idx = k[1..]
                    .parse::<usize>()
                    .with_context(|| anyhow!("Bad text index."))?;
                side_menu[idx].text = v;
            }
            Some('d') => {
                let idx = k[1..]
                    .parse::<usize>()
                    .with_context(|| anyhow!("Bad description index."))?;
                side_menu[idx].description = v;
            }
            _ => Err(anyhow!("Bad prefix."))?,
        }
    }

    customize.side_menu = side_menu
        .into_iter()
        .filter(|menu| !menu.text.is_empty())
        .collect();
    Config::get().customize.save(customize).await?;
    Ok(Redirect::to("/settings").into_response())
}

async fn save_customize_file(path: &Path, data: &[u8]) -> Result<()> {
    let mut file = fs::File::create(path).await?;
    file.write_all(data).await?;
    Ok(())
}

async fn settings_reset_default_ws(session: Session, ws: WebSocketUpgrade) -> ErrorStringResult {
    if !cfg!(debug_assertions) && !session.logged_in().await {
        // close socket immediately
        Err((StatusCode::UNAUTHORIZED, "not logged in"))?;
    }
    Ok(ws
        .on_upgrade(settings_reset_default_socket_handler)
        .into_response())
}

async fn settings_reset_default_socket_handler(mut socket: WebSocket) {
    settings_reset_default(&mut socket).await;
    let _ = socket.send(ws::Message::Close(None)).await;
    while let Some(Ok(_)) = socket.recv().await {}
}

async fn settings_reset_default(socket: &mut WebSocket) {
    let (mut output, _) = socket.split();
    if let Err(e) = stream_command(
        &["reset_default.sh"],
        &CommandOpts {
            // This should not stop halfway
            stream_ignore_output_errors: true,
            ..Default::default()
        },
        &mut output,
    )
    .await
    {
        let _ = output
            .send(ws::Message::text(format!("Error reset default: {e:?}")))
            .await;
        return;
    };
    let _ = output
        .send(ws::Message::text("reset default exited".to_string()))
        .await;
}

async fn settings_get_log() -> ErrorStringResult {
    let output = common::exec_command(&["get_log.sh"]).await?;
    Ok(String::from_utf8_lossy(&output.stdout)
        .to_string()
        .into_response())
}

async fn settings_download_log() -> PageResult {
    let now = Local::now();
    let datetime_str = now.format("%Y%m%d_%H%M%S").to_string();
    let archive_name = format!("armadillo_log_{datetime_str}");
    let tgz_name = format!("{archive_name}.tar.gz");
    let output = common::exec_command(&["archive_log.sh", &archive_name]).await?;

    // Create the response with the archive file content
    let response = Response::builder()
        .header(header::CONTENT_TYPE, "application/octet-stream")
        .header(
            header::CONTENT_DISPOSITION,
            format!("attachment; filename=\"{tgz_name}\""),
        )
        .body(axum::body::Body::from(output.stdout))
        .map_err(|e| anyhow!("Failed to create response: {}", e))?;
    Ok(response.into_response())
}
