// SPDX-License-Identifier: MIT

use async_trait::async_trait;
use parking_lot::{Mutex, MutexGuard};
use std::{
    collections::VecDeque,
    sync::Arc,
    time::{Duration, Instant},
};
use tower_sessions::{
    session::{Id, Record},
    session_store, SessionStore,
};
use tracing::trace;

#[derive(Debug)]
struct StoreItem {
    id: Id,
    record: Record,
    stored_date: Instant,
}

#[derive(Clone, Debug)]
pub struct MemoryStore {
    store: Arc<Mutex<VecDeque<StoreItem>>>,
    lifetime: Duration,
}

impl MemoryStore {
    pub fn new(lifetime: Duration) -> Self {
        MemoryStore {
            store: Mutex::new(VecDeque::new()).into(),
            lifetime,
        }
    }

    fn find(&self, queue: &MutexGuard<VecDeque<StoreItem>>, session_id: &Id) -> Option<Record> {
        for item in queue.iter() {
            if &item.id == session_id {
                return Some(item.record.clone());
            }
        }
        None
    }

    fn is_active(&self, item: &StoreItem) -> bool {
        item.stored_date.elapsed() < self.lifetime
    }

    fn purge_old(&self, queue: &mut MutexGuard<VecDeque<StoreItem>>) {
        while queue.front().is_some_and(|item| !self.is_active(item)) {
            queue.pop_front();
        }
    }
    fn remove(&self, queue: &mut MutexGuard<VecDeque<StoreItem>>, session_id: &Id) {
        queue.retain(|item| !(&item.id == session_id))
    }
}

#[async_trait]
impl SessionStore for MemoryStore {
    async fn save(&self, record: &Record) -> session_store::Result<()> {
        trace!("Saving {:?}", record);
        let mut store = self.store.lock();
        self.purge_old(&mut store);
        self.remove(&mut store, &record.id);
        store.push_back(StoreItem {
            id: record.id,
            record: record.clone(),
            stored_date: Instant::now(),
        });
        trace!("Full queue after save: {:?}", store);
        Ok(())
    }

    async fn create(&self, record: &mut Record) -> session_store::Result<()> {
        trace!("Creating {:?}", record);
        let mut store = self.store.lock();
        self.purge_old(&mut store);
        while self.find(&store, &record.id).is_some() {
            // Session ID collision mitigation.
            record.id = Id::default()
        }
        store.push_back(StoreItem {
            id: record.id,
            record: record.clone(),
            stored_date: Instant::now(),
        });
        trace!("Full queue after save: {:?}", store);
        Ok(())
    }

    async fn load(&self, session_id: &Id) -> session_store::Result<Option<Record>> {
        trace!("Loading {session_id:?}");
        let mut store = self.store.lock();
        self.purge_old(&mut store);
        Ok(self.find(&store, session_id))
    }

    async fn delete(&self, session_id: &Id) -> session_store::Result<()> {
        trace!("Removing {session_id:?}");
        let mut store = self.store.lock();
        self.purge_old(&mut store);
        self.remove(&mut store, session_id);
        Ok(())
    }
}
