// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use std::sync::Arc;
use tokio::net::TcpStream;
use tokio_rustls::{rustls::ServerConfig, server::TlsStream, TlsAcceptor};

use crate::server::args;

pub type Acceptor = TlsAcceptor;

pub async fn tls_init() -> Result<Acceptor> {
    let args = args();
    let cert_data = tokio::fs::read(&args.tls_cert).await?;
    let key_data = tokio::fs::read(&args.tls_key).await?;
    let cert = rustls_pemfile::certs(&mut cert_data.as_ref())
        .flatten()
        .collect::<Vec<_>>();
    let key = rustls_pemfile::private_key(&mut key_data.as_ref())?.context("No key found")?;
    let mut tls_config = ServerConfig::builder()
        .with_no_client_auth()
        .with_single_cert(cert, key)?;
    tls_config.alpn_protocols = vec![b"h2".to_vec(), b"http/1.1".to_vec()];

    Ok(TlsAcceptor::from(Arc::new(tls_config)))
}

pub async fn tls_accept(acceptor: Acceptor, cnx: TcpStream) -> Result<TlsStream<TcpStream>> {
    Ok(acceptor.accept(cnx).await?)
}
