// SPDX-License-Identifier: MIT

use axum::{response::IntoResponse, routing::post, Router};
use serde::Deserialize;

use crate::common::{
    exec_command, exec_command_no_script, CheckAuthRestApi, JsonOrFormOption,
    RestApiPermissionPoweroff, RestApiPermissionReboot, RestApiPermissionSleep,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/reboot", post(restapi_reboot))
        .route("/api/poweroff", post(restapi_poweroff))
        .route("/api/sleep", post(restapi_sleep))
        .route("/api/inhibit_sleep", post(restapi_inhibit_sleep))
}

/// POST "/api/reboot"
/// - Access: reboot
/// - Input: None
/// - Output: None
async fn restapi_reboot(_auth: CheckAuthRestApi<RestApiPermissionReboot>) -> ErrorStringResult {
    exec_command(&["power.sh", "reboot"]).await?;
    Ok(().into_response())
}

/// POST "/api/poweroff"
/// - Access: poweroff
/// - Input: None
/// - Output: None
async fn restapi_poweroff(_auth: CheckAuthRestApi<RestApiPermissionPoweroff>) -> ErrorStringResult {
    exec_command(&["power.sh", "poweroff"]).await?;
    Ok(().into_response())
}

/// POST "/api/sleep"
/// - Access: sleep
/// - Input: None
/// - Output: None, OK sent after wake up if tcp connection not reset
async fn restapi_sleep(_auth: CheckAuthRestApi<RestApiPermissionSleep>) -> ErrorStringResult {
    exec_command(&["power.sh", "sleep"]).await?;
    Ok(().into_response())
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct InhibitSleep {
    inhibit: Option<bool>,
}

/// POST "/api/inhibit_sleep"
/// - Access: sleep
/// - Input: inhibit: optional bool (true if absent)
/// - Output: empty 200 on success, error on error.
///
/// NOTE: power-utils provides two set of sleep control APIs, inibhit_sleep/allow_sleep and
/// disable_sleep/enable_sleep, but we only expose the inhibit variant.
async fn restapi_inhibit_sleep(
    _auth: CheckAuthRestApi<RestApiPermissionSleep>,
    JsonOrFormOption(params): JsonOrFormOption<InhibitSleep>,
) -> ErrorStringResult {
    // inhibit ends when the parent of power-utils stops, so we must not use a wrapper script
    let action = if params.and_then(|p| p.inhibit).unwrap_or(true) {
        "inhibit_sleep"
    } else {
        "allow_sleep"
    };
    exec_command_no_script(&["power-utils", action]).await?;
    Ok(().into_response())
}
