// SPDX-License-Identifier: MIT

use crate::common::{check_auth, exec_command, get_title, HtmlTemplate, Title};
use crate::error::PageResult;
use askama::Template;
use axum::{
    extract::Extension,
    middleware,
    response::IntoResponse,
    routing::{get, post},
    Form, Router,
};
use serde::Deserialize;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

pub fn routes() -> Router {
    Router::new()
        .route("/power", get(power))
        .route("/power_command", post(power_command))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./power")
        }))
}

#[derive(Template)]
#[template(path = "../src/power_management/templates/power.html")]
struct PowerTemplate {}

#[derive(Deserialize)]
struct PowerCommandParam {
    command: String,
}

#[derive(Template)]
#[template(path = "../src/power_management/templates/power_command.html")]
struct PowerCommandTemplate {
    command: String,
}

async fn power(Extension(title): Extension<Title>) -> PageResult {
    let template = PowerTemplate {};
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn power_command(Form(power_command_param): Form<PowerCommandParam>) -> PageResult {
    let command = match power_command_param.command {
        c if c == "reboot" || c == "poweroff" => c,
        _ => Err("invalid command")?,
    };
    exec_command(&["power.sh", command.as_str()]).await?;
    let template = PowerCommandTemplate { command };
    Ok(HtmlTemplate::new("実行", template).into_response())
}
