// SPDX-License-Identifier: MIT

use axum::{
    extract::Path,
    response::IntoResponse,
    routing::{delete, get, post},
    Router,
};

use crate::common::{
    self, json_response, CheckAuthRestApi, IptablesInfo, JsonOrForm, PortForwardingInfo,
    RestApiPermissionNetworkAdmin, RestApiPermissionNetworkView,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/nat", get(rest_nat_list))
        .route("/api/nat/{interface}", post(rest_nat_setup))
        .route("/api/nat/{interface}", delete(rest_nat_delete))
        .route("/api/port_forwarding", get(rest_pf_list))
        .route("/api/port_forwarding", post(rest_pf_setup))
        .route("/api/port_forwarding", delete(rest_pf_delete))
}

/// GET "/api/nat"
/// - Access: Network view
/// - Input: None
/// - Output: List of objects with {interface}, natted interfaces.
async fn rest_nat_list(_auth: CheckAuthRestApi<RestApiPermissionNetworkView>) -> ErrorStringResult {
    let iptables_info = IptablesInfo::get().await?;
    json_response(&iptables_info.nat)
}

/// POST "/api/nat/{interface}"
/// - Access: Network admin
/// - Input: None
/// - Output: None
async fn rest_nat_setup(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(interface): Path<String>,
) -> ErrorStringResult {
    common::exec_command(&["nat_setup.sh", &interface]).await?;
    Ok(().into_response())
}

/// DELETE "/api/nat/{interface}"
/// - Access: Network admin
/// - Input: None
/// - Output: None
async fn rest_nat_delete(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(interface): Path<String>,
) -> ErrorStringResult {
    common::exec_command(&["nat_delete.sh", &interface]).await?;
    Ok(().into_response())
}

/// GET "/api/port_forwarding"
/// - Access: Network view
/// - Input: None
/// - Output: List of objects with {interface, protocol, dport, destination, destination_port}
async fn rest_pf_list(_auth: CheckAuthRestApi<RestApiPermissionNetworkView>) -> ErrorStringResult {
    let iptables_info = IptablesInfo::get().await?;
    json_response(&iptables_info.port_forwarding)
}

/// POST "/api/port_forwarding"
/// - Access: Network admin
/// - Input: interface, protocol, dport, destination, destination_port
/// - Output: None
async fn rest_pf_setup(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    JsonOrForm(params): JsonOrForm<PortForwardingInfo>,
) -> ErrorStringResult {
    common::exec_command(&[
        "port_forwarding_setup.sh",
        &params.interface,
        &params.protocol,
        &params.dport,
        &params.destination,
        &params.destination_port,
    ])
    .await?;
    Ok(().into_response())
}

/// DELETE "/api/nat/{interface}"
/// - Access: Network admin
/// - Input: interface, protocol, dport, destination, destination_port;
///   as per setup or output of list
/// - Output: None
async fn rest_pf_delete(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    JsonOrForm(params): JsonOrForm<PortForwardingInfo>,
) -> ErrorStringResult {
    common::exec_command(&[
        "port_forwarding_delete.sh",
        &params.interface,
        &params.protocol,
        &params.dport,
        &params.destination,
        &params.destination_port,
    ])
    .await?;
    Ok(().into_response())
}
