// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    extract::{Extension, Form},
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use serde::Deserialize;

use crate::common::{
    self, check_auth, get_title, Config, HtmlTemplate, IptablesInfo, NatInfo, PortForwardingInfo,
    Title,
};
use crate::error::PageResult;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize, Debug)]
struct PortForwardingDelParam {
    delete_pf: String,
}

#[derive(Template)]
#[template(path = "../src/nat/templates/nat.html")]
struct NatTemplate {
    nat_list: Vec<NatInfo>,
    pf_list: Vec<PortForwardingInfo>,
    network_interfaces: Vec<String>,
    nat_enabled: bool,
}

pub fn routes() -> Router {
    Router::new()
        .route("/nat", get(nat))
        .route("/nat_setup", post(nat_setup))
        .route("/nat_delete", post(nat_delete))
        .route("/port_forwarding_setup", post(port_forwarding_setup))
        .route("/port_forwarding_delete", post(port_forwarding_delete))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./nat")
        }))
}

async fn nat(Extension(title): Extension<Title>) -> PageResult {
    let iptables_info = IptablesInfo::get().await?;
    let network_interfaces = common::network_interfaces();

    let template = NatTemplate {
        nat_list: iptables_info.nat,
        pf_list: iptables_info.port_forwarding,
        network_interfaces,
        nat_enabled: Config::get().customize.get().nat_enabled,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn nat_setup(Form(nat_param): Form<NatInfo>) -> PageResult {
    if !Config::get().customize.get().nat_enabled {
        return Err("Could not setup nat. This page has been disabled.")?;
    }
    let args = &["nat_setup.sh", &nat_param.interface];
    common::exec_command(args).await?;

    Ok(Redirect::to("/nat").into_response())
}

async fn nat_delete(Form(nat_del_param): Form<NatInfo>) -> PageResult {
    if !Config::get().customize.get().nat_enabled {
        return Err("Could not delete nat. This page has been disabled.")?;
    }
    let args = &["nat_delete.sh", &nat_del_param.interface];
    common::exec_command(args).await?;

    Ok(Redirect::to("/nat").into_response())
}

async fn port_forwarding_setup(Form(pf_param): Form<PortForwardingInfo>) -> PageResult {
    let args = &[
        "port_forwarding_setup.sh",
        &pf_param.interface,
        &pf_param.protocol,
        &pf_param.dport,
        &pf_param.destination,
        &pf_param.destination_port,
    ];
    common::exec_command(args).await?;

    Ok(Redirect::to("/nat").into_response())
}

async fn port_forwarding_delete(Form(pf_del_param): Form<PortForwardingDelParam>) -> PageResult {
    let mut args = vec!["port_forwarding_delete.sh"];
    for param in pf_del_param.delete_pf.split(',') {
        args.push(param)
    }
    common::exec_command(&args).await?;

    Ok(Redirect::to("/nat").into_response())
}
