// SPDX-License-Identifier: MIT

use anyhow::Result;
use axum::{
    extract::Request,
    http::{header, HeaderValue, StatusCode},
    middleware::Next,
    response::{IntoResponse, Response},
    Router,
};
use tower_http::services::ServeDir;
use tower_sessions::{Expiry, SessionManagerLayer};
use tracing::info;

// common subdirs
mod args;
mod common;
mod error;
mod ip_filter;
mod server;
mod session_store;

// pages
mod connection;
mod container;
mod custom_rest;
mod ddns;
mod dhcp;
mod login;
mod nat;
mod nwstatbar;
mod power_management;
mod settings;
mod status;
mod swu;
mod time_settings;
mod top;
mod usb_filter;
mod vpn;
mod wlan;
mod wwan;

#[cfg(debug_assertions)]
const STATIC_RESOURCE_DIR: &str = "static";

#[cfg(not(debug_assertions))]
const STATIC_RESOURCE_DIR: &str = "/usr/share/abos-web/static";

const STATIC_CUSTOMIZE_RESOURCE_DIR: &str = "/etc/atmark/abos_web/customize/static";

async fn fallback_404() -> (StatusCode, &'static str) {
    (StatusCode::NOT_FOUND, "Page not found")
}

async fn add_security_headers(req: Request, next: Next) -> Response {
    let mut response = next.run(req).await;
    let headers = response.headers_mut();

    // Content Security Policy with modern XSS protection (replaces X-XSS-Protection)
    headers.insert(
        header::CONTENT_SECURITY_POLICY,
        HeaderValue::from_static(
            "default-src 'self'; \
            script-src 'self' 'unsafe-inline'; \
            style-src 'self' 'unsafe-inline'; \
            img-src 'self'; \
            connect-src 'self'; \
            font-src 'self'; \
            object-src 'none'; \
            media-src 'self'; \
            frame-src 'none'; \
            frame-ancestors 'none'; \
            base-uri 'self'; \
            form-action 'self'; \
            upgrade-insecure-requests;",
        ),
    );
    // CSP directive explanations:
    // default-src 'self': Only allow same-origin resources by default
    // script-src 'self' 'unsafe-inline': Allow same-origin + inline JavaScript
    //   Note: 'unsafe-inline' should be migrated to nonce or hash-based in the future
    // style-src 'self' 'unsafe-inline': Allow same-origin + inline CSS
    // img-src 'self': Restrict images to same-origin only
    // connect-src 'self': Restrict HTTP/HTTPS connections to same-origin only
    // object-src 'none': Completely prohibit <object>, <embed> tags
    // frame-src 'none': Prohibit iframe content loading
    // frame-ancestors 'none': Prohibit iframe embedding by other sites
    // form-action 'self': Restrict form submissions to same-origin only (CSRF protection)
    // upgrade-insecure-requests: Automatically upgrade HTTP to HTTPS

    // Referrer-Policy: Explicitly set for clarity (modern browsers default to this)
    headers.insert(
        header::REFERRER_POLICY,
        HeaderValue::from_static("strict-origin-when-cross-origin"),
    );

    headers.insert(
        header::X_CONTENT_TYPE_OPTIONS,
        HeaderValue::from_static("nosniff"),
    );
    // X-Content-Type-Options: Disable browser MIME type sniffing
    // Strictly follow server-set Content-Type, prevent content-based detection
    // Prevents malicious file script execution

    response
}

// Add text body to some common errors (for curl users who might not
// notice the error)
pub async fn enhance_errors(req: Request, next: Next) -> Response {
    let response = next.run(req).await;

    match response.status() {
        StatusCode::METHOD_NOT_ALLOWED => {
            (StatusCode::METHOD_NOT_ALLOWED, "Method not allowed").into_response()
        }
        _ => response,
    }
}

#[tokio::main]
async fn main() -> Result<()> {
    args::args_init();
    common::password_throttle_init();

    let session_store = session_store::MemoryStore::new(std::time::Duration::from_secs(12 * 3600));

    let session_service = SessionManagerLayer::new(session_store)
        .with_secure(true)
        .with_same_site(tower_sessions::cookie::SameSite::Strict)
        .with_http_only(true)
        .with_expiry(Expiry::OnInactivity(time::Duration::hours(12)));

    common::Variant::load().await?;
    let variant = common::Variant::get();
    info!(
        "Loaded variant: {}  - {}",
        variant.name, variant.description
    );

    common::Config::load().await?;

    let app = Router::new()
        .merge(connection::routes())
        .merge(container::routes())
        .merge(custom_rest::routes())
        .merge(dhcp::routes())
        .merge(login::routes())
        .merge(nat::routes())
        .merge(nwstatbar::routes())
        .merge(power_management::routes())
        .merge(settings::routes())
        .merge(status::routes())
        .merge(swu::routes())
        .merge(time_settings::routes())
        .merge(top::routes())
        .merge(vpn::routes())
        .merge(wlan::routes())
        .merge(wwan::routes())
        .merge(usb_filter::routes())
        .merge(ddns::routes())
        .nest_service(
            "/static",
            ServeDir::new(STATIC_CUSTOMIZE_RESOURCE_DIR)
                .fallback(ServeDir::new(STATIC_RESOURCE_DIR)),
        )
        .layer(session_service)
        .route_layer(axum::middleware::from_fn(ip_filter::ipaddr_filter))
        .fallback(fallback_404)
        .layer(axum::middleware::from_fn(enhance_errors))
        .layer(axum::middleware::from_fn(add_security_headers));

    server::run_server(app).await?;
    Ok(())
}
