// SPDX-License-Identifier: MIT

use axum::{routing::post, Router};
use serde::Deserialize;

use crate::common::{self, CheckAuthRestApi, JsonOrForm, RestApiPermissionAdmin};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new().route("/api/password", post(rest_register_password))
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct RestPasswordParam {
    password: String,
    hashed: Option<bool>,
}

/// POST "/api/password"
/// - Access: admin
/// - Input: password=[pass or hash], optional hashed=false to allow plain text
/// - Output: None
async fn rest_register_password(
    _auth: CheckAuthRestApi<RestApiPermissionAdmin>,
    JsonOrForm(param): JsonOrForm<RestPasswordParam>,
) -> ErrorStringResult<()> {
    if param.password.is_empty() {
        Err("Password must not be empty")?
    }
    let mut args = vec!["register_password.sh"];
    if param.hashed.unwrap_or(true) {
        args.push("-e");
    }
    common::exec_command_stdin(&args, &param.password).await?;
    Ok(())
}
