// SPDX-License-Identifier: MIT

use askama::Template;
use axum::{
    http::StatusCode,
    response::{IntoResponse, Response},
    Json,
};
use serde_json::json;
use tower_sessions::session;

use crate::common::HtmlTemplate;

#[derive(Template)]
#[template(path = "../src/error/templates/error.html")]
struct ErrorTemplate {
    message: String,
}

/// We can't implement IntoResponse (external trait) onto anyhow::Error (external type)
/// as both are external, so implement our own error page type that will have the trait
pub struct ErrorPage {
    pub message: String,
}

/// an alias for Result<Response, ErrorPage> as that will be used a lot
pub type PageResult<T = Response, E = ErrorPage> = core::result::Result<T, E>;

/// error_with_message for backwards compatibility until it is no longer used
pub fn error_with_message(message: String) -> Response {
    let template = ErrorTemplate { message };
    let mut response = HtmlTemplate::new("エラー", template).into_response();
    *response.status_mut() = StatusCode::INTERNAL_SERVER_ERROR;
    response
}
/// Implementing IntoResponse allows us to use the type on either end of a Result:
/// axum will automatically call err.into_response() when Err(err) is returned
impl IntoResponse for ErrorPage {
    //fn into_response(self) -> Response<http_body::combinators::UnsyncBoxBody<axum::body::Bytes, axum::Error>> {
    fn into_response(self) -> Response {
        error_with_message(self.message)
    }
}

/// This allows converting from &str errors as used in common/common.rs
impl From<&str> for ErrorPage {
    fn from(source: &str) -> Self {
        ErrorPage {
            message: source.to_string(),
        }
    }
}

/// This allows using ? with anyhow errors
impl From<anyhow::Error> for ErrorPage {
    fn from(source: anyhow::Error) -> Self {
        ErrorPage {
            message: format!("{source:#}"),
        }
    }
}

impl From<session::Error> for ErrorPage {
    fn from(source: session::Error) -> Self {
        ErrorPage {
            message: source.to_string(),
        }
    }
}

/// implement a simpler error type for XHR data that don't return full HTML
pub struct ErrorString {
    error: String,
    code: StatusCode,
}

pub type ErrorStringResult<T = Response, E = ErrorString> = core::result::Result<T, E>;

impl IntoResponse for ErrorString {
    fn into_response(self) -> Response {
        (self.code, Json(json!({"error": self.error}))).into_response()
    }
}

impl From<(StatusCode, &str)> for ErrorString {
    fn from((code, source): (StatusCode, &str)) -> Self {
        ErrorString {
            error: source.to_string(),
            code,
        }
    }
}

impl From<&str> for ErrorString {
    fn from(source: &str) -> Self {
        ErrorString {
            error: source.to_string(),
            code: StatusCode::INTERNAL_SERVER_ERROR,
        }
    }
}

impl From<String> for ErrorString {
    fn from(source: String) -> Self {
        ErrorString {
            error: source,
            code: StatusCode::INTERNAL_SERVER_ERROR,
        }
    }
}

impl From<anyhow::Error> for ErrorString {
    fn from(source: anyhow::Error) -> Self {
        ErrorString {
            error: format!("{source:#}"),
            code: StatusCode::INTERNAL_SERVER_ERROR,
        }
    }
}
