// SPDX-License-Identifier: MIT

use axum::{
    extract::Path,
    response::IntoResponse,
    routing::{delete, get, post},
    Router,
};
use serde::Deserialize;

use crate::common::{
    self, dhcp_list, json_response, CheckAuthRestApi, JsonOrForm, RestApiPermissionNetworkAdmin,
    RestApiPermissionNetworkView,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/dhcp", get(rest_dhcp_list))
        .route("/api/dhcp/{interface}", post(rest_dhcp_setup))
        .route("/api/dhcp/{interface}", delete(rest_dhcp_delete))
}

/// GET "/api/dhcp"
/// - Access: Network view
/// - Input: None
/// - Output: list of objects (interface, ip_addr (if set), start_addr,
///   end_addr, lease_time}
async fn rest_dhcp_list(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
) -> ErrorStringResult {
    let list = dhcp_list().await?;
    json_response(&list)
}

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct DhcpSetupParam {
    start_addr: String,
    end_addr: String,
    lease_time: Option<String>,
}

/// POST "/api/dhcp/{interface}"
/// - Access: Network admin
/// - Input: start_addr, end_addr, optional lease_time (default 1h)
/// - Output: None
async fn rest_dhcp_setup(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(interface): Path<String>,
    JsonOrForm(params): JsonOrForm<DhcpSetupParam>,
) -> ErrorStringResult {
    common::exec_command(&[
        "dhcp_setup.sh",
        &params.start_addr,
        &params.end_addr,
        params.lease_time.as_deref().unwrap_or("3600"),
        &interface,
    ])
    .await?;
    Ok(().into_response())
}

/// DELETE "/api/dhcp/{interface}"
/// - Access: Network admin
/// - Input: None
/// - Output: None
async fn rest_dhcp_delete(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(interface): Path<String>,
) -> ErrorStringResult {
    common::exec_command(&["dhcp_delete.sh", &interface]).await?;
    Ok(().into_response())
}
