// SPDX-License-Identifier: MIT

use anyhow::Context;
use askama::Template;
use axum::{
    extract::{Extension, Form},
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use serde::Deserialize;

use crate::common;
use crate::common::{
    check_auth, dhcp_info, filters, get_title,
    networkmanager::{nmconvec, NMCon},
    DhcpInfo, HtmlTemplate, Title,
};
use crate::error::PageResult;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize)]
struct DhcpParam {
    interface: String,
    start_addr: String,
    end_addr: String,
    lease_time: String,
}

#[derive(Deserialize)]
struct DhcpDelParam {
    interface: String,
}

struct DhcpCandidate {
    interface: String,
    ip: String,
}

#[derive(Template)]
#[template(path = "../src/dhcp/templates/dhcp.html")]
struct DhcpTemplate {
    candidates: Vec<DhcpCandidate>,
    dhcp_infos: Vec<DhcpInfo>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/dhcp", get(dhcp))
        .route("/dhcp_setup", post(dhcp_setup))
        .route("/dhcp_delete", post(dhcp_delete))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./dhcp")
        }))
}

async fn dhcp(Extension(title): Extension<Title>) -> PageResult {
    let cons = nmconvec().await?;
    let mut candidates = vec![];
    let mut dhcp_infos = vec![];
    for con in cons {
        // cons values don't have enough info, will add a helper to enrich later
        // but for now fetch manually...
        let Ok(con) = NMCon::from_id(&con.uuid).await else {
            continue;
        };
        // ignore interfaces not in manual mode
        if con
            .prop("ipv4.method")
            .map(|v| v != "manual")
            .unwrap_or(true)
        {
            continue;
        }
        if con.device.is_none() {
            continue;
        }
        let interface = con
            .device
            .as_ref()
            .context("no interface after checking none")?
            .as_str();
        // ignore podman interfaces
        if interface.starts_with("podman") {
            continue;
        }
        if let Ok(dhcp_info) = dhcp_info(interface).await {
            dhcp_infos.push(dhcp_info);
        } else {
            // candidate
            if let Some(ip) = con.prop("IP4.ADDRESS[1]") {
                candidates.push(DhcpCandidate {
                    interface: interface.to_string(),
                    ip: ip.to_string(),
                });
            }
        }
    }

    candidates.sort_unstable_by(|a, b| a.interface.cmp(&b.interface));
    dhcp_infos.sort_unstable_by(|a, b| a.interface.cmp(&b.interface));

    let template = DhcpTemplate {
        candidates,
        dhcp_infos,
    };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn dhcp_setup(Form(dhcp_param): Form<DhcpParam>) -> PageResult {
    let args = &[
        "dhcp_setup.sh",
        &dhcp_param.start_addr,
        &dhcp_param.end_addr,
        &dhcp_param.lease_time,
        &dhcp_param.interface,
    ];
    common::exec_command(args).await?;
    Ok(Redirect::to("/dhcp").into_response())
}

async fn dhcp_delete(Form(dhcp_del_param): Form<DhcpDelParam>) -> PageResult {
    let args = &["dhcp_delete.sh", &dhcp_del_param.interface];
    common::exec_command(args).await?;
    Ok(Redirect::to("/dhcp").into_response())
}
