// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Result};
use askama::Template;
use axum::{
    body::Bytes,
    extract::Extension,
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use axum_typed_multipart::{FieldData, TryFromMultipart, TypedMultipart};
use std::collections::HashMap;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

use crate::common::{self, check_auth, get_title, HtmlTemplate, Title};
use crate::error::{ErrorStringResult, PageResult};

#[derive(TryFromMultipart)]
struct DdnsParam {
    ddns_service: String,
    hostname: String,
    username: Option<String>,
    password: String,
    proxied: Option<String>,
    interval: String,
    conf: FieldData<Bytes>,
}

#[derive(Template)]
#[template(path = "../src/ddns/templates/ddns.html")]
struct DdnsTemplate<'a> {
    ddns_info: HashMap<&'a str, &'a str>,
}

pub fn routes() -> Router {
    Router::new()
        .route("/ddns", get(ddns))
        .route("/ddns_setup", post(ddns_setup))
        .route("/ddns_delete", post(ddns_delete))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./ddns")
        }))
}

async fn ddns(Extension(title): Extension<Title>) -> PageResult {
    let args = &["ddns_info.sh"];
    let output = common::exec_command(args).await?;
    let info_output = String::from_utf8_lossy(&output.stdout);
    let ddns_info = extract_ddns_info(&info_output);

    let template = DdnsTemplate { ddns_info };
    Ok(HtmlTemplate::new(title.0, template).into_response())
}

async fn ddns_setup(TypedMultipart(ddns_param): TypedMultipart<DdnsParam>) -> ErrorStringResult {
    // 現在の設定を削除
    common::exec_command(&["ddns_delete.sh"]).await?;

    // 設定スクリプト引数の基本部分
    let mut args = vec!["ddns_setup.sh", &ddns_param.ddns_service];

    // ddns_service によって処理分岐
    match ddns_param.ddns_service.as_str() {
        "file" => {
            // 設定ファイルアップロード処理
            let conf_path = "/etc/atmark/abos_web/inadyn.conf";
            tokio::fs::write(conf_path, ddns_param.conf.contents)
                .await
                .map_err(|e| anyhow!("Failed to write config file: {}", e))?;
        }
        _ => {
            // 通常のサービス設定処理
            args.push(&ddns_param.hostname);
            args.push("--password");
            args.push(&ddns_param.password);
            args.push("--interval");
            args.push(&ddns_param.interval);

            // オプション引数の追加
            if let Some(user) = &ddns_param.username {
                if !user.is_empty() {
                    args.extend_from_slice(&["--user", user]);
                }
            }

            if let Some(proxied) = &ddns_param.proxied {
                if !proxied.is_empty() {
                    args.extend_from_slice(&["--proxied", proxied]);
                }
            }
        }
    }

    // 設定スクリプト実行
    common::exec_command(&args).await?;

    Ok(Redirect::to("/ddns").into_response())
}

async fn ddns_delete() -> PageResult {
    let args = &["ddns_delete.sh"];
    common::exec_command(args).await?;

    Ok(Redirect::to("/ddns").into_response())
}

// ヘルパー関数
fn extract_ddns_info(info_output: &str) -> HashMap<&str, &str> {
    let mut info = HashMap::new();

    for line in info_output.lines() {
        if let Some((key, value)) = line.split_once(" = ") {
            match key {
                "provider1" => info.insert("service1", value),
                "hostname1" => info.insert("hostname1", value),
                "period" => info.insert("interval", value),
                "provider2" => info.insert("service2", value),
                "hostname2" => info.insert("hostname2", value),
                _ => None,
            };
        }
    }

    info
}
