// SPDX-License-Identifier: MIT

use axum::{extract::Path, response::IntoResponse, routing::post, Router};
use serde::Deserialize;

use crate::common::{
    json_stream_command, CheckAuthRestApi, CommandOpts, JsonOrFormOption, RestApiPermissionCustom,
};
use crate::error::ErrorString;

pub fn routes() -> Router {
    Router::new().route("/api/custom/{script}", post(restapi_custom))
}

// note: debug build runs through ssh so the path is the same
const REST_CUSTOM_SCRIPTS_PATH: &str = "/etc/atmark/abos_web/customize_rest";

#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct CustomRestParam {
    #[serde(default)]
    args: Vec<String>,
    root: Option<bool>,
}

/// POST "/api/custom/{script}"
/// - Access: Custom
/// - Input: list of 'args' to pass to script, 'root' argument to run as root.
/// - Output: lines
async fn restapi_custom(
    _auth: CheckAuthRestApi<RestApiPermissionCustom>,
    Path(script): Path<String>,
    JsonOrFormOption(params): JsonOrFormOption<CustomRestParam>,
) -> impl IntoResponse {
    // forbid scripts with slashes or odd chars through whitelist
    if !script
        .chars()
        .all(|c| c.is_ascii_alphanumeric() || "._-".contains(c))
    {
        return ErrorString::from(
            "script must only contain alphanumeric characters or the following characters: ._-",
        )
        .into_response();
    }
    let prog = format!("{REST_CUSTOM_SCRIPTS_PATH}/{script}");
    let mut args = vec![prog];
    let mut as_root = false;

    if let Some(mut params) = params {
        args.append(&mut params.args);
        as_root = params.root.unwrap_or(false);
    }

    json_stream_command(
        args,
        &CommandOpts {
            is_script: false,
            as_root,
            ..Default::default()
        },
        None,
    )
    .into_response()
}
