// SPDX-License-Identifier: MIT

use axum::{
    extract::Path,
    response::IntoResponse,
    routing::{delete, get, patch, post},
    Router,
};
use serde::Serialize;
use std::collections::HashMap;

use crate::common::{
    json_response,
    networkmanager::{nmcon_act, nmcon_mod, nmconvec, NMCon},
    CheckAuthRestApi, JsonOrForm, RestApiPermissionNetworkAdmin, RestApiPermissionNetworkView,
};
use crate::error::ErrorStringResult;

pub fn routes() -> Router {
    Router::new()
        .route("/api/connections", get(rest_connections_list))
        .route("/api/connections/{connection}", get(rest_connections_get))
        .route(
            "/api/connections/{connection}",
            patch(rest_connections_patch),
        )
        .route(
            "/api/connections/{connection}",
            delete(rest_connections_delete),
        )
        .route(
            "/api/connections/{connection}/up",
            post(rest_connections_up),
        )
        .route(
            "/api/connections/{connection}/down",
            post(rest_connections_down),
        )
}

#[derive(Serialize)]
struct RestConnectionList {
    connections: Vec<NMCon>,
}

/// GET "/api/connections"
/// - Access: network view
/// - Input: None
/// - Output: RestConnectionList == json object with 'connections', an array of objects
///   with {uuid, name, state, ctype, optional device}
async fn rest_connections_list(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
) -> ErrorStringResult {
    let connections = nmconvec().await?;
    json_response(&RestConnectionList { connections })
}

/// GET "/api/connections/{connection}"
/// - Access: network view
/// - Input: None
/// - Output: NMCon == same as list + props
async fn rest_connections_get(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkView>,
    Path(connection): Path<String>,
) -> ErrorStringResult {
    json_response(&NMCon::from_id(&connection).await?)
}

/// PATCH "/api/connections/{connection}"
/// - Access: network admin
/// - Input: fields to modifiy
/// - Output: None
async fn rest_connections_patch(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(connection): Path<String>,
    JsonOrForm(props): JsonOrForm<HashMap<String, String>>,
) -> ErrorStringResult {
    if !props.is_empty() {
        nmcon_mod(connection, props).await?;
    }
    Ok(().into_response())
}

/// POST "/api/connections/{connection}/up"
/// - Access: network admin
/// - Input: None
/// - Output: None (on success)
async fn rest_connections_up(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(connection): Path<String>,
) -> ErrorStringResult {
    nmcon_act("up", &connection).await?;
    Ok(().into_response())
}

/// POST "/api/connections/{connection}/down"
/// - Access: network admin
/// - Input: None
/// - Output: None (on success)
async fn rest_connections_down(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(connection): Path<String>,
) -> ErrorStringResult {
    nmcon_act("down", &connection).await?;
    Ok(().into_response())
}

/// DELETE "/api/connections/{connection}"
/// - Access: network admin
/// - Input: None
/// - Output: None (on success)
async fn rest_connections_delete(
    _auth: CheckAuthRestApi<RestApiPermissionNetworkAdmin>,
    Path(connection): Path<String>,
) -> ErrorStringResult {
    nmcon_act("delete", &connection).await?;
    Ok(().into_response())
}
