// SPDX-License-Identifier: MIT

use anyhow::{Context, Result};
use askama::Template;
use axum::{
    extract::{Extension, Form},
    middleware,
    response::{IntoResponse, Redirect},
    routing::{get, post},
    Router,
};
use serde::Deserialize;
use std::collections::HashMap;

use crate::common::{
    check_auth, filters, get_title,
    networkmanager::{nmcon_act, nmcon_mod, nmconvec, NMCon},
    HtmlTemplate, Title,
};
use crate::error::PageResult;

#[cfg(feature = "restapi")]
mod restapi;
#[cfg(not(feature = "restapi"))]
mod restapi {
    pub fn routes() -> axum::Router {
        axum::Router::new()
    }
}

#[derive(Deserialize)]
struct UuidParam {
    con_uuid: String,
    return_url: Option<String>,
}

#[derive(Template)]
#[template(path = "../src/connection/templates/connection.html")]
struct ConnectionTemplate {
    connections: Vec<NMCon>,
}

#[derive(Template)]
#[template(path = "../src/connection/templates/connection_setup.html")]
struct ConnectionSetupTemplate<'a> {
    con_uuid: String,
    // postprocessing hash tables too heavily in template is limited,
    // we generate html here directly
    props_html: Vec<String>,
    return_url: &'a str,
}

pub fn routes() -> Router {
    Router::new()
        .route("/connection", get(connection))
        .route("/connection_setup", get(connection_setup))
        .route("/connection_save", post(connection_save))
        .route("/connection_up", post(connection_up))
        .route("/connection_down", post(connection_down))
        .route("/connection_delete", post(connection_delete))
        .route_layer(middleware::from_fn(check_auth))
        .merge(restapi::routes())
        .route_layer(middleware::from_fn(|request, next| {
            get_title(request, next, "./connection")
        }))
}

async fn connection(Extension(title): Extension<Title>) -> PageResult {
    let mut connections: Vec<NMCon> = nmconvec()
        .await?
        .into_iter()
        .map(|mut conn| {
            let ctype = match conn.ctype.as_str() {
                "802-3-ethernet" => "ethernet".to_string(),
                "802-11-wireless" => "wireless".to_string(),
                _ => conn.ctype,
            };
            conn.ctype = ctype;
            conn
        })
        .collect();
    connections.sort_unstable_by(|a, b| a.name.cmp(&b.name));
    let template = ConnectionTemplate { connections };

    Ok(HtmlTemplate::new(title.0, template).into_response())
}

// template for each input field.
// It might be simpler ultimately to just define classes here and have javascript
// hook in events on DOMContentLoaded, but this works...
#[derive(Default)]
struct ConnectionSetupField<'a> {
    field: &'a str,
    name: &'a str,
    class: Option<&'a str>,
    choices: Vec<&'a str>,
    // only for no choice
    pattern: Option<&'a str>,
    tooltip: Option<&'a str>,
}

#[derive(Template)]
#[template(
    source = r#"
    <label
        {%- if let Some(class) = csf.class -%}
          {#- comments like the one below are used to control spaces, as ~ keeps a newline... -#}
          {##} class="{{ class }}"
        {%- endif -%}
        {##} for="{{ csf.field }}">
      {%- if csf.name.is_empty() -%}
        {{ csf.field }}
      {%- else -%}
        {{ csf.name }} ({{ csf.field }})
      {%- endif -%}
    </label>
    {%- if let Some(tooltip) = csf.tooltip %}
      {%- if let Some(class) = csf.class %}
    <div class="{{ class }}">
      {%- endif %}
    <div class="flex">
    {%- endif %}
    {%- if csf.choices.is_empty() %}
      <input type="text" id="{{ csf.field }}" name="{{ csf.field }}" value="{{ value }}" {# -#}
          class="optional_input
          {%- if let Some(class) = csf.class -%}
            {##} {{ class }}
          {%- endif -%}
        "
        {%- if let Some(pattern) = csf.pattern -%}
          {##} pattern="{{ pattern }}"
        {%- endif -%}
        {##} />
    {%- else %}
      <select id="{{ csf.field }}" name="{{ csf.field }}" {# -#}
        class="optional_input
          {%- if let Some(class) = csf.class -%}
            {##} {{ class }}
          {%- endif -%}
        ">
          <option value="" selected disabled hidden>{{ value }}</option>
           {%- for choice in csf.choices %}
              <option value="{{ choice }}">{{ choice }}</option>
           {%- endfor %}
        </select>
    {%- endif %}
    {%- if let Some(tooltip) = csf.tooltip %}
       <span class="tooltip
           {%- if let Some(class) = csf.class -%}
             {##} {{ class }}
           {%- endif -%}
           " data-tooltip="{{ tooltip }}">?</span>
    </div>
    {%- if let Some(class) = csf.class %}
    </div>
    {%- endif %}
    {%- endif -%}
"#,
    ext = "html"
)]
struct InputFieldTemplate<'a> {
    csf: ConnectionSetupField<'a>,
    value: String,
}

fn csf<'a>(field: &'a str, name: &'a str) -> ConnectionSetupField<'a> {
    ConnectionSetupField {
        field,
        name,
        ..ConnectionSetupField::default()
    }
}
fn csf_class<'a>(field: &'a str, name: &'a str, class: &'a str) -> ConnectionSetupField<'a> {
    ConnectionSetupField {
        field,
        name,
        class: Some(class),
        ..ConnectionSetupField::default()
    }
}
fn csf_class_pattern<'a>(
    field: &'a str,
    name: &'a str,
    class: &'a str,
    pattern: &'a str,
    tooltip: &'a str,
) -> ConnectionSetupField<'a> {
    ConnectionSetupField {
        field,
        name,
        class: Some(class),
        pattern: Some(pattern),
        tooltip: Some(tooltip),
        ..ConnectionSetupField::default()
    }
}
fn csf_choice<'a>(
    field: &'a str,
    name: &'a str,
    choices: Vec<&'a str>,
) -> ConnectionSetupField<'a> {
    ConnectionSetupField {
        field,
        name,
        choices,
        ..ConnectionSetupField::default()
    }
}
fn csf_choice_class<'a>(
    field: &'a str,
    name: &'a str,
    choices: Vec<&'a str>,
    class: &'a str,
) -> ConnectionSetupField<'a> {
    ConnectionSetupField {
        field,
        name,
        class: Some(class),
        choices,
        ..ConnectionSetupField::default()
    }
}

fn input_from_csf(csf: ConnectionSetupField, value: String) -> Result<String> {
    Ok(InputFieldTemplate { csf, value }.render()?)
}

async fn connection_setup(Form(param): Form<UuidParam>) -> PageResult {
    let con = NMCon::from_id(&param.con_uuid).await?;
    let (con_uuid, mut props) = (con.uuid, con.props.context("con did not have props")?);
    let mut props_html = Vec::<String>::new();
    // common patterns
    let pattern_ip = r"\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}";
    let pattern_ips = format!(r"{pattern_ip}([ ,]{pattern_ip})*");
    let pattern_subnet = format!(r"{pattern_ip}/\d{{1,2}}");
    let pattern_subnets = format!(r"{pattern_subnet}([ ,]{pattern_subnet})*");
    // from nm-settings-nmcli(5): "ip[/prefix] [next-hop] [metric] [attribute=val]...[,ip[/prefix]...]"
    let pattern_route =
        format!(r"{pattern_ip}(/\d{{1,2}})?( {pattern_ip})?( \d+)?( [^ =]*=[^ =]*)?");
    let pattern_routes = format!(r"{pattern_route}(,{pattern_route})*");
    let pattern_ip6 = r"[0-9a-fA-F:]*";
    let pattern_ips6 = format!(r"{pattern_ip6}([ ,]{pattern_ip6})*");
    let pattern_subnet6 = format!(r"{pattern_ip6}/\d{{1,3}}");
    let pattern_subnets6 = format!(r"{pattern_subnet6}([ ,]{pattern_subnet6})*");
    let pattern_route6 =
        format!(r"{pattern_ip6}(/\d{{1,3}})?( {pattern_ip6})?( \d+)?( [^ =]*=[^ =]*)?");
    let pattern_routes6 = format!(r"{pattern_route6}(,{pattern_route6})*");
    let ipv4_manual_classes = if props
        .get("ipv4.method")
        .map(|mode| mode == "manual")
        .unwrap_or(false)
    {
        "ipv4_manual"
    } else {
        "hiddenfield ipv4_manual"
    };
    let ipv6_manual_classes = if props
        .get("ipv6.method")
        .map(|mode| mode == "manual")
        .unwrap_or(false)
    {
        "ipv6_manual"
    } else {
        "hiddenfield ipv6_manual"
    };
    let fixed_fields = vec![
        csf("connection.id", "接続名"),
        csf("connection.interface-name", "インターフェース"),
        // only listed for wlan
        csf("802-11-wireless.ssid", "接続先のネットワーク名"),
        csf("802-11-wireless-security.psk", "WiFi パスワード"),
        // only listed for wwan
        csf("gsm.apn", "APN"),
        csf("gsm.username", "GSM ユーザー名"),
        csf("gsm.password", "GSM パスワード"),
        // XXX do we want to show gsm.network-id ppp.refuse-eap ppp.refuse-chap
        // ppp.refuse-mschap ppp.refuse-mschapv2 ppp.refuse-pap ?
        // common settings
        csf_choice_class(
            "ipv4.method",
            "IPv4 取得モード",
            vec![
                "auto",
                "manual",
                "disabled",
                "link-local",
                //"shared", not allowing shared as not supported for ABOS
            ],
            "toggle_ipv4_manual",
        ),
        csf_class_pattern(
            "ipv4.addresses",
            "IPv4 アドレス",
            ipv4_manual_classes,
            &pattern_subnets,
            "ip address with netmask e.g. 192.168.1.2/24",
        ),
        csf_class_pattern(
            "ipv4.gateway",
            "IPv4 ゲートウェイ",
            ipv4_manual_classes,
            pattern_ip,
            "ip address e.g. 192.168.1.1",
        ),
        csf_class_pattern(
            "ipv4.dns",
            "IPv4 DNS",
            ipv4_manual_classes,
            &pattern_ips,
            "ip addresses, e.g. 8.8.8.8,1.1.1.1",
        ),
        csf_class_pattern(
            "ipv4.routes",
            "IPv4 スタティックルート",
            ipv4_manual_classes,
            &pattern_routes,
            "static routes: ip[/prefix] [next-hop] [metric] [attribute=val]...[,ip[/prefix]...]",
        ),
        csf("ipv4.route-metric", "IPv4 ルーティングメトリック"),
        csf_choice_class(
            "ipv6.method",
            "IPv6 取得モード",
            vec![
                "auto",
                "manual",
                "disabled",
                "ignore",
                "link-local",
                "dhcp",
                //"shared", not allowing shared as not supported for ABOS
            ],
            "toggle_ipv6_manual",
        ),
        csf_class_pattern(
            "ipv6.addresses",
            "IPv6 アドレス",
            ipv6_manual_classes,
            &pattern_subnets6,
            "ip address with netmask e.g. fdd4:8638:ad20::1/48",
        ),
        csf_class_pattern(
            "ipv6.gateway",
            "IPv6 ゲートウェイ",
            ipv6_manual_classes,
            pattern_ip6,
            "ip address e.g. fe80::1",
        ),
        csf_class_pattern(
            "ipv6.dns",
            "IPv6 DNS",
            ipv6_manual_classes,
            &pattern_ips6,
            "ip addresses e.g. 2001:4860:4860::8888,2606:4700:4700::1111",
        ),
        csf_class_pattern(
            "ipv6.routes",
            "IPv6 スタティックルート",
            ipv6_manual_classes,
            &pattern_routes6,
            "static routes: ip[/prefix] [next-hop] [metric] [attribute=val]...[,ip[/prefix]...]",
        ),
        csf("ipv6.route-metric", "IPv6 ルーティングメトリック"),
        csf_choice("connection.autoconnect", "自動コネクト", vec!["yes", "no"]),
    ];

    // remove special props:
    // uuid is already handled as con_uuid
    let _ = props.remove("connection.uuid");
    // type/timestamp/read-only cannot be changed
    let _ = props.remove("connection.type");
    let _ = props.remove("connection.timestamp");
    let _ = props.remove("connection.read-only");

    for csf in fixed_fields {
        if let Some(prop) = props.remove(csf.field) {
            props_html.push(input_from_csf(csf, prop)?)
        }
    }

    // hardcode toggle button for details between fixed fields
    // and prop details
    props_html.push(
        r#"<button type="button" class="submit toggle_details">詳細を表示</button>"#.to_string(),
    );

    for (key, value) in props {
        if let 'A'..='Z' = key.chars().next().unwrap_or(' ') {
            continue;
        }
        props_html.push(input_from_csf(
            csf_class(&key, "", "prop_details hiddenfield"),
            value,
        )?)
    }

    let return_url = param.return_url.as_deref().unwrap_or("/connection");
    let template = ConnectionSetupTemplate {
        con_uuid,
        props_html,
        return_url,
    };
    Ok(HtmlTemplate::new("接続設定", template).into_response())
}

async fn connection_save(Form(mut props): Form<HashMap<String, String>>) -> PageResult {
    let uuid = props.remove("uuid").context("parameters had no uuid")?;
    let return_url = props.remove("return_url");
    let return_url = return_url.as_deref().unwrap_or("/connection");
    if !props.is_empty() {
        nmcon_mod(uuid, props).await?;
    }

    Ok(Redirect::to(return_url).into_response())
}

async fn connection_up(Form(param): Form<UuidParam>) -> PageResult {
    nmcon_act("up", &param.con_uuid).await?;

    Ok(Redirect::to("/connection").into_response())
}

async fn connection_down(Form(param): Form<UuidParam>) -> PageResult {
    nmcon_act("down", &param.con_uuid).await?;

    Ok(Redirect::to("/connection").into_response())
}

async fn connection_delete(Form(param): Form<UuidParam>) -> PageResult {
    nmcon_act("delete", &param.con_uuid).await?;

    Ok(Redirect::to("/connection").into_response())
}
