use anyhow::{Context, Result};
use once_cell::sync::OnceCell;
use serde::Deserialize;
use std::env;
use std::sync::Arc;
use tokio::fs;

use crate::common::Customize;

#[cfg(debug_assertions)]
const STATIC_VARIANT_DIR: &str = "variant/";

#[cfg(not(debug_assertions))]
const STATIC_VARIANT_DIR: &str = "/usr/share/abos-web/variant/";

static VARIANT: OnceCell<Variant> = OnceCell::new();

#[derive(Clone, Debug)]
pub struct Variant {
    inner: Arc<VariantInner>,
}

#[derive(Clone, Deserialize, Debug)]
pub struct VariantInner {
    pub name: String,
    pub description: String,
    pub customize: Customize,
}

impl Variant {
    pub async fn load() -> Result<()> {
        let variant_type = env::var("ABOSWEB_VARIANT").unwrap_or_else(|_| "standard".to_string());
        let variant_file = format!("{STATIC_VARIANT_DIR}{variant_type}.json");

        let variant_str = fs::read_to_string(&variant_file)
            .await
            .with_context(|| format!("Failed to read variant file {variant_file}"))?;
        let variant_inner = serde_json::from_str::<VariantInner>(&variant_str)
            .with_context(|| format!("could not parse variant file {variant_file}"))?;

        VARIANT
            .set(Variant {
                inner: variant_inner.into(),
            })
            .expect("Load should only be called once");
        Ok(())
    }

    pub fn get() -> &'static VariantInner {
        &VARIANT.get().expect("Variant is not initialized").inner
    }
}
