// SPDX: CC-BY-SA
// https://stackoverflow.com/questions/31101915/how-to-implement-trim-for-vecu8
// https://github.com/rust-lang/rust/pull/112818/files

pub trait SliceExt {
    fn trim(&self) -> &Self;
    fn split_once<F>(&self, pred: F) -> Option<(&Self, &Self)>
    where
        F: FnMut(&u8) -> bool;
}

impl SliceExt for [u8] {
    fn trim(&self) -> &Self {
        fn is_whitespace(c: &u8) -> bool {
            *c == b'\t' || *c == b' '
        }

        fn is_not_whitespace(c: &u8) -> bool {
            !is_whitespace(c)
        }

        if let Some(first) = self.iter().position(is_not_whitespace) {
            if let Some(last) = self.iter().rposition(is_not_whitespace) {
                &self[first..last + 1]
            } else {
                unreachable!();
            }
        } else {
            &[]
        }
    }

    fn split_once<F>(&self, pred: F) -> Option<(&Self, &Self)>
    where
        F: FnMut(&u8) -> bool,
    {
        let index = self.iter().position(pred)?;
        Some((self.get(..index)?, self.get(index + 1..)?))
    }
}
