// SPDX-License-Identifier: MIT

use anyhow::{Context, Error, Result};
use std::io::ErrorKind;
use std::path::Path;
use tokio::fs;
use tracing::debug;
use uuid::Uuid;

use crate::common::restapi::{parse_token, serialize_token, SerializeTokenOptions};
use crate::common::{RestApiPermission, RestApiToken};

#[cfg(not(test))]
use crate::common::exec_command;

fn check_valid_token(token_id: &str) -> Result<()> {
    if token_id.contains(['/', '.']) {
        Err(Error::msg("Invalid token string"))?
    }
    Ok(())
}

async fn load_token(path: &Path) -> Result<RestApiToken> {
    if path.extension().unwrap_or_default() != "token" {
        return Err(Error::msg("Ignoring file not ending in .token"));
    }
    let token_id = path
        .file_stem()
        .and_then(|stem| stem.to_str())
        .context("Invalid token string")?
        .to_string();
    match fs::read_to_string(&path).await {
        Ok(content) => parse_token(token_id, &content),
        Err(e) if e.kind() == ErrorKind::NotFound => Err(Error::msg("No such token")),
        Err(e) => Err(e)?,
    }
}
pub async fn get_token(tokens_path: &Path, token_id: &str) -> Result<RestApiToken> {
    check_valid_token(token_id)?;
    load_token(&tokens_path.join(token_id).with_extension("token")).await
}

pub async fn list_tokens(tokens_path: &Path) -> Result<Vec<RestApiToken>> {
    let mut dir = match fs::read_dir(tokens_path).await {
        Ok(dir) => dir,
        Err(e) if e.kind() == ErrorKind::NotFound => return Ok(vec![]),
        Err(e) => Err(e)?,
    };
    let mut tokens = Vec::new();

    while let Some(dirent) = dir.next_entry().await? {
        match load_token(&dirent.path()).await {
            Ok(token) => tokens.push(token),
            Err(e) => debug!("Skipped {:?}: {:?}", dirent.file_name(), e),
        }
    }
    Ok(tokens)
}

/// write token to disk (overwrites if present)
pub async fn write_token(tokens_path: &Path, token: &RestApiToken) -> Result<()> {
    check_valid_token(&token.token_id)?;
    // recursive mode ignores error if directory already exists
    fs::DirBuilder::new()
        .recursive(true)
        .mode(0o700)
        .create(tokens_path)
        .await?;

    debug!("Writing token to {tokens_path:?}: {token:?}");
    fs::write(
        tokens_path.join(&token.token_id).with_extension("token"),
        serialize_token(
            token,
            &SerializeTokenOptions {
                include_token_id: false,
            },
        )?,
    )
    .await?;

    // persist token directory after changes except for tests
    #[cfg(not(test))]
    exec_command(&["persist_rest_tokens.sh"]).await?;

    Ok(())
}

/// create token
pub async fn create_token(
    tokens_path: &Path,
    permissions: Vec<RestApiPermission>,
) -> Result<RestApiToken> {
    // generate random UUIDs until a free one is found.
    // (Could probably just error out on collision but looping is cheap enough...)
    let token_id = loop {
        let token_id = Uuid::new_v4().to_string();
        if get_token(tokens_path, &token_id).await.is_err() {
            break token_id;
        }
    };
    let token = RestApiToken {
        token_id,
        permissions,
    };
    write_token(tokens_path, &token).await?;
    Ok(token)
}

pub async fn delete_token(tokens_path: &Path, token_id: &str) -> Result<()> {
    check_valid_token(token_id)?;
    match fs::remove_file(tokens_path.join(token_id).with_extension("token")).await {
        Err(e) if e.kind() == ErrorKind::NotFound => Err(Error::msg("No such token"))?,
        Err(e) => Err(e)?,
        _ => (),
    };

    // persist token directory after changes except for tests
    #[cfg(not(test))]
    exec_command(&["persist_rest_tokens.sh"]).await?;

    Ok(())
}

#[cfg(test)]
mod tests {
    use anyhow::Result;
    use std::path::PathBuf;
    use tempfile::TempDir;

    use super::*;
    use crate::common::{RestApiPermission, RestApiToken};

    const TEST_TOKENS_PATH: &str = "tests/data/tokens";
    const TEST_TOKENS_PATH_NODIR: &str = "tests/data/tokens_nosuchdir";

    #[test]
    fn test_parse_token() -> Result<()> {
        let admin_perm = vec![RestApiPermission::Admin];
        let admin_token = RestApiToken {
            token_id: "0".to_string(),
            permissions: admin_perm,
        };

        // valid json example
        let token = parse_token(
            "0".to_string(),
            r#"
{ "token": "test1234", "permissions": ["Admin"] }
"#,
        )?;
        assert_eq!(&token, &admin_token);

        // invalid json error
        assert!(parse_token("".into(), "").is_err());

        Ok(())
    }

    #[tokio::test]
    async fn test_list_tokens() -> Result<()> {
        // (sorted by token_id)
        let tokens_ref = vec![
            RestApiToken {
                token_id: "admin_token".to_string(),
                permissions: vec![RestApiPermission::Admin],
            },
            RestApiToken {
                token_id: "default".to_string(),
                permissions: vec![RestApiPermission::Admin],
            },
            RestApiToken {
                token_id: "ignore_id".to_string(),
                permissions: vec![RestApiPermission::Admin],
            },
            RestApiToken {
                token_id: "multiperm".to_string(),
                permissions: vec![RestApiPermission::Admin, RestApiPermission::NetworkAdmin],
            },
        ];

        // valid tokens directory
        let mut tokens = list_tokens(&PathBuf::from(TEST_TOKENS_PATH)).await?;
        assert_eq!(tokens.len(), 4);
        tokens.sort_unstable_by(|a, b| a.token_id.cmp(&b.token_id));
        assert_eq!(tokens, tokens_ref);

        for token_ref in tokens_ref {
            let token = get_token(&PathBuf::from(TEST_TOKENS_PATH), &token_ref.token_id).await?;
            assert_eq!(token, token_ref);
        }

        // no dir = empty list
        assert_eq!(
            list_tokens(&PathBuf::from(TEST_TOKENS_PATH_NODIR))
                .await
                .unwrap(),
            vec![]
        );
        // some invalid examples
        assert!(get_token(&PathBuf::from(TEST_TOKENS_PATH), "no such file")
            .await
            .is_err());

        Ok(())
    }

    #[tokio::test]
    async fn test_write_delete_token() -> Result<()> {
        let perm = vec![RestApiPermission::Admin, RestApiPermission::SwuView];

        let tmp_dir = TempDir::new()?;
        let dir_path = tmp_dir.path();
        let mut token = create_token(dir_path, perm).await?;

        let loaded_token = get_token(dir_path, &token.token_id).await?;
        assert_eq!(token, loaded_token);

        token.permissions.push(RestApiPermission::NetworkAdmin);
        write_token(dir_path, &token).await?;
        let loaded_token = get_token(dir_path, &token.token_id).await?;
        assert_eq!(token, loaded_token);

        delete_token(dir_path, &token.token_id).await?;
        assert!(get_token(dir_path, &token.token_id).await.is_err());

        Ok(())
    }
}
