// SPDX-License-Identifier: MIT

use anyhow::Result;
use axum::{extract::FromRequestParts, http::request::Parts, http::StatusCode};
use std::marker::PhantomData;
use std::time::Duration;
use tokio::time::Instant;
use tower_sessions::Session;
use tracing::{debug, trace};

use crate::common::auth_header::{AuthHeader, AuthHeaders};
use crate::common::password::password_throttle_lock;
use crate::common::restapi::{check_token, permissions::ToRestApiPermission};
use crate::common::{check_password, LoggedIn, RestApiPermission};
use crate::error::ErrorString;

// Extractor can only take parameters as types, and it needs to be stored in
// the struct as PhantomData for the compiler.
// See https://github.com/tokio-rs/axum/discussions/1533
pub struct CheckAuthRestApi<P> {
    _perm: PhantomData<P>,
}

impl<S, P> FromRequestParts<S> for CheckAuthRestApi<P>
where
    S: Send + Sync,
    P: Send + ToRestApiPermission,
{
    type Rejection = ErrorString;

    // validate auth if ok for rest call:
    // - if token was provided, check it matches requested auth (RestApiPermission*)
    // or admin.
    // - if auth basic was provided, check password matches
    // - if session exists, check if logged in
    async fn from_request_parts(parts: &mut Parts, state: &S) -> Result<Self, Self::Rejection> {
        let auth_ok = CheckAuthRestApi { _perm: PhantomData };

        match AuthHeaders::from_parts(parts).next() {
            Some(AuthHeader::Basic(user, pass)) => {
                trace!("got header basic (user {})", user);
                if check_password(&pass).await.is_err() {
                    Err((StatusCode::UNAUTHORIZED, "Invalid password"))?;
                }
                return Ok(auth_ok);
            }
            Some(AuthHeader::Bearer(token)) => {
                trace!("got header bearer {}", token);
                if let Err(msg) = check_restapi_token(token, P::to_permission()).await {
                    Err((StatusCode::UNAUTHORIZED, msg))?;
                }
                return Ok(auth_ok);
            }
            _ => (),
        }

        // already logged in user is also allowed
        if let Ok(session) = Session::from_request_parts(parts, state).await {
            if session.logged_in().await {
                return Ok(auth_ok);
            }
        }
        Err((StatusCode::UNAUTHORIZED, "No auth found"))?
    }
}

async fn check_restapi_token(token_id: &str, perm: RestApiPermission) -> Result<(), &'static str> {
    let _lock = password_throttle_lock().await;
    let start_time = Instant::now();
    if let Err(e) = check_token(token_id, perm).await {
        debug!("Bad token {token_id}: {e:?}");
        tokio::time::sleep_until(start_time + Duration::from_secs(2)).await;
        return Err("Invalid token or insufficient access rights");
    }
    Ok(())
}
