// SPDX-License-Identifier: MIT

use anyhow::{Context, Error, Result};
use serde::{Deserialize, Serialize};
use tracing::warn;

use crate::common::exec_command;

#[derive(Debug)]
pub struct IptablesInfo {
    pub nat: Vec<NatInfo>,
    pub port_forwarding: Vec<PortForwardingInfo>,
}

#[derive(Debug, Deserialize, Serialize)]
#[serde(deny_unknown_fields)]
pub struct NatInfo {
    pub interface: String,
}

// when deserializing, default protocol to tcp
fn default_proto_tcp() -> String {
    "tcp".to_string()
}

#[derive(Debug, PartialEq, Eq, Deserialize, Serialize)]
#[serde(deny_unknown_fields)]
pub struct PortForwardingInfo {
    pub interface: String,
    #[serde(default = "default_proto_tcp")]
    pub protocol: String,
    pub dport: String,
    pub destination: String,
    pub destination_port: String,
}

enum AnyInfo {
    Nat(NatInfo),
    PortForwarding(PortForwardingInfo),
}

impl AnyInfo {
    fn from_iptables_line(rule: &str) -> Result<Self> {
        // should probably use nom for that... But here goes anyway.
        if rule.contains("MASQUERADE") {
            //-A POSTROUTING -o eth0 -m comment --comment abos-web-nat -j MASQUERADE
            let trail = rule
                .strip_prefix("-A POSTROUTING -o ")
                .context("masquerade without postrouting prefix")?;
            let iface = trail
                .split_once(' ')
                .context("postrouting interface without suffix")?
                .0;
            return Ok(AnyInfo::Nat(NatInfo {
                interface: iface.to_string(),
            }));
        }
        if rule.contains("DNAT") {
            //-A PREROUTING -i eth0 -p tcp -m tcp --dport 2134 -m comment --comment abos-web-forwarding -j DNAT --to-destination 1.2.3.4:4213
            let mut interface: Option<String> = None;
            let mut protocol: Option<String> = None;
            let mut dport: Option<String> = None;
            let mut destination: Option<String> = None;
            let mut destination_port: Option<String> = None;
            // need to collect once to use chunks...
            for chunk in rule.split(' ').collect::<Vec<&str>>().chunks(2) {
                match chunk[0] {
                    "-i" => interface = Some(chunk[1].to_string()),
                    "-p" => protocol = Some(chunk[1].to_string()),
                    "--dport" => dport = Some(chunk[1].to_string()),
                    "--to-destination" => {
                        let dest_split =
                            chunk[1].split_once(':').context("destination without :")?;
                        destination = Some(dest_split.0.to_string());
                        destination_port = Some(dest_split.1.to_string())
                    }
                    _ => (), // ignore
                }
            }
            return Ok(AnyInfo::PortForwarding(PortForwardingInfo {
                interface: interface.context("DNAT with no interface")?,
                protocol: protocol.context("DNAT with no protocol")?,
                dport: dport.context("DNAT with no dport")?,
                destination: destination.context("DNAT with no destination")?,
                destination_port: destination_port.context("DNAT with no destination_port")?,
            }));
        }
        Err(Error::msg("Neither nat nor port forwarding"))
    }
}

impl IptablesInfo {
    pub fn from_iptables_show(output: &str) -> Result<Self> {
        let mut nat = Vec::<NatInfo>::new();
        let mut port_forwarding = Vec::<PortForwardingInfo>::new();

        let rules = output
            .trim()
            .split('\n')
            .filter(|r| r.contains(r#"--comment abos-web"#));
        for rule in rules {
            match AnyInfo::from_iptables_line(rule) {
                Ok(AnyInfo::Nat(nat_info)) => nat.push(nat_info),
                Ok(AnyInfo::PortForwarding(pf_info)) => port_forwarding.push(pf_info),
                Err(e) => warn!(
                    "Could not parse iptables line with abos-web comment: {}: {}",
                    rule, e
                ),
            }
        }

        Ok(IptablesInfo {
            nat,
            port_forwarding,
        })
    }
    pub async fn get() -> Result<Self> {
        let args = &["iptables_info.sh"];
        IptablesInfo::from_iptables_show(&String::from_utf8_lossy(
            &exec_command(args).await?.stdout,
        ))
    }
}

#[cfg(test)]
mod tests {
    use crate::common::iptables::{IptablesInfo, PortForwardingInfo};
    use anyhow::Result;
    #[test]
    fn test_parse_iptables() -> Result<()> {
        let info = IptablesInfo::from_iptables_show(
            r#"
-P PREROUTING ACCEPT
-P INPUT ACCEPT
-P OUTPUT ACCEPT
-P POSTROUTING ACCEPT
-N NETAVARK-1D8721804F16F
-N NETAVARK-DN-1D8721804F16F
-N NETAVARK-HOSTPORT-DNAT
-N NETAVARK-HOSTPORT-MASQ
-N NETAVARK-HOSTPORT-SETMARK
-A PREROUTING -m addrtype --dst-type LOCAL -j NETAVARK-HOSTPORT-DNAT
-A PREROUTING -i eth0 -p tcp -m tcp --dport 2134 -m comment --comment abos-web-forwarding -j DNAT --to-destination 1.2.3.4:4213
-A PREROUTING -i eth0 -p tcp -m tcp --dport 5341 -m comment --comment abos-web-forwarding -j DNAT --to-destination 1.2.3.4:1111
-A OUTPUT -m addrtype --dst-type LOCAL -j NETAVARK-HOSTPORT-DNAT
-A POSTROUTING -j NETAVARK-HOSTPORT-MASQ
-A POSTROUTING -s 10.88.0.0/16 -j NETAVARK-1D8721804F16F
-A POSTROUTING -o eth0 -m comment --comment abos-web-nat -j MASQUERADE
-A NETAVARK-1D8721804F16F -d 10.88.0.0/16 -j ACCEPT
-A NETAVARK-1D8721804F16F ! -d 224.0.0.0/4 -j MASQUERADE
-A NETAVARK-DN-1D8721804F16F -s 10.88.0.0/16 -p tcp -m tcp --dport 80 -j NETAVARK-HOSTPORT-SETMARK
-A NETAVARK-DN-1D8721804F16F -s 127.0.0.1/32 -p tcp -m tcp --dport 80 -j NETAVARK-HOSTPORT-SETMARK
-A NETAVARK-DN-1D8721804F16F -p tcp -m tcp --dport 80 -j DNAT --to-destination 10.88.0.2:80
-A NETAVARK-HOSTPORT-DNAT -d 10.88.0.1/32 -p udp -m udp --dport 53 -j DNAT --to-destination 10.88.0.1:153
-A NETAVARK-HOSTPORT-DNAT -p tcp -m tcp --dport 80 -m comment --comment "dnat name: podman id: d29b712e5c05859ad8942f5f6a31ed12f370f0cc7e15462b3b3d5b8517c4023e" -j NETAVARK-DN-1D8721804F16F
-A NETAVARK-HOSTPORT-MASQ -m comment --comment "netavark portfw masq mark" -m mark --mark 0x2000/0x2000 -j MASQUERADE
-A NETAVARK-HOSTPORT-SETMARK -j MARK --set-xmark 0x2000/0x2000
            "#,
        )?;
        assert_eq!(info.nat.len(), 1);
        assert_eq!(info.nat[0].interface, "eth0");
        assert_eq!(info.port_forwarding.len(), 2);
        assert_eq!(
            info.port_forwarding[0],
            PortForwardingInfo {
                interface: "eth0".to_string(),
                protocol: "tcp".to_string(),
                dport: "2134".to_string(),
                destination: "1.2.3.4".to_string(),
                destination_port: "4213".to_string(),
            }
        );
        assert_eq!(
            info.port_forwarding[1],
            PortForwardingInfo {
                interface: "eth0".to_string(),
                protocol: "tcp".to_string(),
                dport: "5341".to_string(),
                destination: "1.2.3.4".to_string(),
                destination_port: "1111".to_string(),
            }
        );
        Ok(())
    }
}
