// SPDX-License-Identifier: MIT

use anyhow::{anyhow, Context, Result};
use serde::Serialize;

use crate::common::exec_command;

#[derive(Serialize, Debug, PartialEq)]
pub struct DhcpInfo {
    pub interface: String,
    pub ip_addr: Option<String>,
    pub start_addr: String,
    pub end_addr: String,
    pub lease_time: String,
}

fn parse_dhcp_info(line: &str) -> Option<DhcpInfo> {
    /*
     * in order:
     *  - interface name e.g. br_ap
     *  - ip addr with netmask e.g. 192.168.1.1/24 (can be empty)
     *  - DHCP start addr
     *  - DHCP end addr
     *  - DHCP lease time
     */
    let mut elems = line.split(',');
    Some(DhcpInfo {
        interface: elems.next()?.to_string(),
        ip_addr: elems.next().and_then(|s| {
            if s.is_empty() {
                None
            } else {
                Some(s.to_string())
            }
        }),
        start_addr: elems.next()?.to_string(),
        end_addr: elems.next()?.to_string(),
        lease_time: elems.next()?.to_string(),
    })
}

pub async fn dhcp_list() -> Result<Vec<DhcpInfo>> {
    let output = exec_command(&["dhcp_info.sh"]).await?;
    let stdout = String::from_utf8_lossy(&output.stdout);
    Ok(stdout.lines().filter_map(parse_dhcp_info).collect())
}

pub async fn dhcp_info(interface: &str) -> Result<DhcpInfo> {
    let output = exec_command(&["dhcp_info.sh", interface]).await?;
    let stdout = String::from_utf8_lossy(&output.stdout);
    if stdout.is_empty() {
        Err(anyhow!("no info available for {}", interface))
    } else {
        parse_dhcp_info(&stdout).with_context(|| format!("Could not parse DHCP info: {stdout}"))
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_parse_dhcp_info() {
        let info = parse_dhcp_info("br_ap,192.168.1.1/24,192.168.1.10,192.168.1.20,3600").unwrap();
        assert_eq!(
            &info,
            &DhcpInfo {
                interface: "br_ap".to_string(),
                ip_addr: Some("192.168.1.1/24".to_string()),
                start_addr: "192.168.1.10".to_string(),
                end_addr: "192.168.1.20".to_string(),
                lease_time: "3600".to_string(),
            }
        );

        let info = parse_dhcp_info("br_ap,,192.168.1.10,192.168.1.20,3600").unwrap();
        assert_eq!(
            &info,
            &DhcpInfo {
                interface: "br_ap".to_string(),
                ip_addr: None,
                start_addr: "192.168.1.10".to_string(),
                end_addr: "192.168.1.20".to_string(),
                lease_time: "3600".to_string(),
            }
        );
    }
}
