// SPDX-License-Identifier: MIT

use crate::common;
use anyhow::Result;
use parking_lot::Mutex;
use serde::{Deserialize, Serialize};
use std::sync::Arc;
use tokio::fs;
use tracing::warn;

const STATIC_CUSTOMIZE_JSON: &str = "/etc/atmark/abos_web/customize.json";

#[derive(Clone, Deserialize, Serialize, Debug)]
pub struct SideMenu {
    pub href_path: String,
    pub text: String,
    pub icon_path: String,
    pub description: String,
}

#[derive(Clone, Deserialize, Serialize, Debug)]
pub struct Customize {
    pub header_title: String,
    pub model_number: Option<String>,
    pub side_menu: Vec<SideMenu>,
    pub color: String,
    #[serde(default = "default_true")]
    pub wlan_client: bool,
    #[serde(default = "default_true")]
    pub nat_enabled: bool,
    #[serde(default = "default_true")]
    pub tokens_enabled: bool,
}

impl Customize {
    pub fn get_page_title(&self, href: &str) -> Option<String> {
        self.side_menu
            .iter()
            .find(|menu| menu.href_path == href)
            .filter(|menu| !menu.text.is_empty())
            .map(|menu| menu.text.clone())
    }
}

impl Default for Customize {
    fn default() -> Self {
        let variant = &common::Variant::get();
        variant.customize.clone()
    }
}

#[derive(Clone, Debug)]
pub struct CustomizeConfig {
    // Arc<Mutex: Mutex cannot be used in structs that use Clone Trait,
    // so it must be wrapped in Arc.
    // Arc<Customize: This is required to shallow copy the contents of Mutex.
    pub customize: Arc<Mutex<Arc<Customize>>>,
}

impl CustomizeConfig {
    pub async fn load() -> Self {
        let customize = match fs::read_to_string(STATIC_CUSTOMIZE_JSON).await {
            Ok(contents) => serde_json::from_str(&contents).unwrap_or_else(|e| {
                warn!("parse customize.json failed: {:?}", e);
                Customize::default()
            }),
            // No such file, use default silently
            Err(e) if e.kind() == std::io::ErrorKind::NotFound => Customize::default(),
            Err(e) => {
                warn!("read customize.json failed: {:?}", e);
                Customize::default()
            }
        };
        CustomizeConfig {
            customize: Arc::new(Mutex::new(Arc::new(customize))),
        }
    }

    pub async fn save(&self, new_customize: Customize) -> Result<()> {
        let serialized = serde_json::to_string(&new_customize)?;
        common::exec_command_stdin(&["persist_customize.sh"], &serialized).await?;
        *self.customize.lock() = Arc::new(new_customize);
        Ok(())
    }

    pub fn get(&self) -> Arc<Customize> {
        self.customize.lock().clone()
    }

    pub fn get_clone(&self) -> Customize {
        Customize::clone(&self.customize.lock())
    }

    pub fn reset(&self) {
        *self.customize.lock() = Arc::new(Customize::default());
    }
}

fn default_true() -> bool {
    true
}
