#!/bin/sh -e
# SPDX-License-Identifier: MIT

CONF_TEMP_DIR="$1"
SETTING_NAME="$2"
AUTH_TYPE="$3"
VPN_CONF_FILE="$4"

CONF_DIR="/etc/atmark/abos_web/openvpn"
USERPASS_FILE="userpass.txt"
KEY_PASS_FILE="keypass.txt"
VPN_INFO_FILE="vpn_info.txt"
CONTAINER_IMAGE_NAME="alpine_openvpn"
CONTAINER_NAME="abos_web_openvpn"
CONTAINER_CONF_FILE="/etc/atmark/containers/abos_web_openvpn.conf"
UP_SCRIPT="/usr/libexec/abos-web/openvpn/up.sh"
DOWN_SCRIPT="/usr/libexec/abos-web/openvpn/down.sh"
DNS_FILE="/etc/dnsmasq.d/openvpn_dns.conf"

trap catch ERR

catch() {
	if openvpn --version > /dev/null 2>&1; then
		if nmcli con show "${VPN_CONF_FILE%.ovpn}" >/dev/null 2>&1; then
			nmcli con del "${VPN_CONF_FILE%.ovpn}"
			persist_file -d "${VPN_CONF_FILE%.ovpn}" || :
		fi
	else
		if podman container exists "$CONTAINER_NAME"; then
			podman stop "$CONTAINER_NAME" || :
			podman rm "$CONTAINER_NAME" || :
		fi
		rm -f "$CONTAINER_CONF_FILE"
		persist_file -d "$CONTAINER_CONF_FILE" || :
		rm -f "$DNS_FILE"
		persist_file -d "$DNS_FILE" || :
		rc-service dnsmasq restart || :

		if [ "$podman_stop" = 1 ]; then
			podman_start -a
		fi
	fi
	rm -rf "$CONF_DIR"
	persist_file -dr "$CONF_DIR" || :
}

check_filename() {
	case "$1" in
		*[!a-zA-Z0-9=_.,:-]*)
			echo "Invalid file name: $1" >&2
			exit 1 ;;
	esac
}

check_filename "$VPN_CONF_FILE"

while [ $# -gt 0 ];
do
	case "$1" in
		"--user")
			shift
			VPNUSER="$1" ;;
		"--password")
			shift
			PASSWORD="$1" ;;
		"--cert")
			shift
			CERT_FILE="$1"
			check_filename "$CERT_FILE" ;;
		"--key")
			shift
			KEY_FILE="$1"
			check_filename "$KEY_FILE";;
		"--askpass")
			shift
			KEY_PASS="$1" ;;
		"--quiet")
			QUIET="1" ;;
		*)
			;;
	esac
	shift
done

if openvpn --version > /dev/null 2>&1; then
	# if openvpn command is installed, use NetworkManager.
	# delete existing connections.
	CON_NAME=$(nmcli -t -f type,name c | awk -F: '$1 == "vpn" { print $2 }')
	if [ -n "$CON_NAME" ]; then
		CON_FILE="/etc/NetworkManager/system-connections/$CON_NAME.nmconnection"
		if nmcli con show "$CON_NAME" >/dev/null 2>&1; then
			nmcli con del "$CON_NAME"
			persist_file -d "$CON_FILE" || :
		fi
	fi

	cat > "$CONF_DIR/$VPN_INFO_FILE" <<EOF
$SETTING_NAME
EOF

	opt=""
	if [ "$AUTH_TYPE" = "userpass" ]; then
		opt="$opt +vpn.data password-flags=0 +vpn.data username=$VPNUSER vpn.secrets password=$PASSWORD"
	else
		if [ -n "$CERT_FILE" ]; then
			cat >> "$CONF_DIR/$VPN_CONF_FILE" <<EOF
<cert>
$(cat "$CONF_DIR/$CERT_FILE")
</cert>
EOF
			rm "$CONF_DIR/$CERT_FILE"
		fi
		if [ -n "$KEY_FILE" ]; then
			cat >> "$CONF_DIR/$VPN_CONF_FILE" <<EOF
<key>
$(cat "$CONF_DIR/$KEY_FILE")
</key>
EOF
			rm "$CONF_DIR/$KEY_FILE"
		fi
		if [ -n "$KEY_PASS" ]; then
			opt="$opt +vpn.data cert-pass-flags=0 vpn.secrets cert-pass=atmark123"
		fi
	fi

	nmcli connection import type openvpn file "$CONF_DIR/$VPN_CONF_FILE"
	if [ -n "$opt" ]; then
		nmcli connection mod "${VPN_CONF_FILE%.ovpn}" $opt
	fi
	nmcli connection up "${VPN_CONF_FILE%.ovpn}"
	rm "$CONF_DIR/$VPN_CONF_FILE"
	persist_file -r /etc/NetworkManager/system-connections
	persist_file -r "$CONF_DIR"

	/usr/libexec/abos-web/nat_setup.sh tun0
else
	# if openvpn command is no installed, use openvpn container.
	if ! [ "$(podman image ls -q "$CONTAINER_IMAGE_NAME")" ]; then
		podman_stop=1
		podman stop -a
		# wait for containers to be fully stopped
		podman ps --format '{{.ID}}' \
			| timeout 20s xargs -r podman wait > /dev/null
		abos-ctrl podman-rw build -t "$CONTAINER_IMAGE_NAME":latest - << 'EOF'
From docker.io/alpine:latest
RUN apk add --no-cache openvpn
EOF
		podman_stop=0
		podman_start -a
	fi

	# Cleanup any previous config if any
	rm -f "$CONF_DIR"
	if [ -n "$KEY_FILE" ]; then
		chmod 400 "$CONF_TEMP_DIR/$KEY_FILE"
	fi
	mv "$CONF_TEMP_DIR" "$CONF_DIR"

	opt=""
	if [ "$AUTH_TYPE" = "userpass" ]; then
		touch "$CONF_DIR/$USERPASS_FILE"
		chmod 400 "$CONF_DIR/$USERPASS_FILE"
		cat > "$CONF_DIR/$USERPASS_FILE" <<EOF
$VPNUSER
$PASSWORD
EOF
		opt="$opt --auth-user-pass $USERPASS_FILE"
	else
		if [ -n "$CERT_FILE" ]; then
			opt="$opt --cert $CERT_FILE"
		fi
		if [ -n "$KEY_FILE" ]; then
			opt="$opt --key $KEY_FILE"
		fi
		if [ -n "$KEY_PASS" ]; then
			touch "$CONF_DIR/$KEY_PASS_FILE"
			chmod 400 "$CONF_DIR/$KEY_PASS_FILE"
			cat > "$CONF_DIR/$KEY_PASS_FILE" <<EOF
$KEY_PASS
EOF
			opt="$opt --askpass $KEY_PASS_FILE"
		fi
	fi

	if grep -q "^cipher AES-128-CBC" "$CONF_DIR/$VPN_CONF_FILE"; then
		opt="$opt --data-ciphers AES-128-CBC"
	fi

	cat > "$DNS_FILE" <<EOF
# resolv-file's directory must exist when dnsmasq starts. We already create
# /run/NetworkMamanger for normal dns operations, so use the same directory
resolv-file=/run/NetworkManager/openvpn_resolv.conf
strict-order
EOF

	opt="$opt --script-security 2 --up $UP_SCRIPT --down $DOWN_SCRIPT"

	cat > "$CONTAINER_CONF_FILE" <<EOF
set_image alpine_openvpn
set_network host
add_devices /dev/net/tun
add_volumes /etc/atmark/abos_web/openvpn
add_volumes /run/NetworkManager
add_volumes /usr/libexec/abos-web/openvpn
add_args -w /etc/atmark/abos_web/openvpn
add_args --cap-add=NET_ADMIN

# replace with a more useful command
set_command openvpn --config $VPN_CONF_FILE $opt
EOF

	cat > "$CONF_DIR/$VPN_INFO_FILE" <<EOF
$SETTING_NAME
EOF

	rc-service dnsmasq restart
	podman_start "$CONTAINER_NAME"
	persist_file -r "$CONF_DIR"
	persist_file "$CONTAINER_CONF_FILE"
	persist_file "$DNS_FILE"
	# When the container is stopped, podman logs fails
	# and the container is deleted by trap.
	# So release the trap here.
	trap - ERR
	if [ -z "$QUIET" ]; then
		"$(dirname "$0")/podman.sh" logs_follow "$CONTAINER_NAME"
	fi
fi
