#!/bin/sh
# SPDX-License-Identifier: MIT

error() {
	printf "%s\n" "$@" >&2
	exit 1
}

tmpdir_create() {
	local hint="$1"
	TMPDIR=$(mktemp -d "/var/tmp/swupdate-abosweb-$hint.XXXXXX") \
		|| error "Could not create temporary directory for swupdate"
}

tmpdir_cleanup() {
	if [ -n "$TMPDIR" ]; then
		find "$TMPDIR" -xdev -delete 2>/dev/null
		unset TMPDIR
	fi
}

main() {
	# updates triggered through this are always interactive, so allow rollback
	export SW_ALLOW_ROLLBACK=1

	# fail early if already rebooting
	if [ -e /run/swupdate_rebooting ]; then
		echo "An update was already installed, aborting" >&2
		exit 1
	fi

	mode="$1"
	shift

	case "$mode" in
	stdin|url) "swupdate_$mode" "$@";;
	*)
		error "Usage: swupdate.sh [stdin|url <url> [<url>...]]"
		;;
	esac
}

swupdate_stdin() {
	local rc
	tmpdir_create upload
	TMPDIR="$TMPDIR" swupdate -i /proc/self/fd/0
	rc=$?
	tmpdir_cleanup
	return $rc
}

swupdate_url() {
	local rc best=1

	# try all urls given
	for url; do
		echo "Downloading $url..."
		tmpdir_create url
		TMPDIR="$TMPDIR" swupdate -d "-u $url"
		rc=$?
		tmpdir_cleanup
		[ "$rc" = 0 ] && best=0
		# we're done, don't bother with more
		[ -e /run/swupdate_rebooting ] && exit $best

	done
	# in case of multiple URLs
	return $best
}

main "$@"
