#!/bin/sh -e
# SPDX-License-Identifier: MIT

DDNS_SERVICE="$1"  # no-ip, dyndns, cloudflare など
HOSTNAME="$2"	  # 更新対象のホスト名

ABOSWEB_INADYN_CONF="/etc/atmark/abos_web/inadyn.conf"
INADYN_CONF="/etc/inadyn/inadyn.conf"
INADYN_CONFD_CONF="/etc/conf.d/inadyn"

trap catch ERR

catch() {
	local rc="$?"
	set +e

	persist_file -R /etc/runlevels/default/inadyn \
		"$INADYN_CONF" \
		"$INADYN_CONFD_CONF" \

	if [ -e "/etc/runlevels/default/inadyn" ]; then
		rc-service inadyn restart
	fi

	rm -f "$ABOSWEB_INADYN_CONF"

	echo "Could not set up DDNS." >&2
	exit $rc
}

while [ $# -gt 0 ];
do
	case "$1" in
		"--user")
			shift
			DDNS_USER="$1" ;;
		"--password")
			shift
			DDNS_PASSWORD="$1" ;;
		"--proxied")
			shift
			DDNS_PROXIED="$1" ;;
		"--interval")
			shift
			UPDATE_INTERVAL="$1" ;;
		*)
			;;
	esac
	shift
done

# デフォルト値設定
UPDATE_INTERVAL="${UPDATE_INTERVAL:-300}"  # 5分間隔

if [ "$DDNS_SERVICE" != "file" ]; then
	# inadyn設定ファイル生成
	cat > "$INADYN_CONF" <<EOF
# inadyn configuration file
period = $UPDATE_INTERVAL
# DDNS service configuration
EOF

	case "$DDNS_SERVICE" in
		"no-ip"|"noip")
			cat >> "$INADYN_CONF" <<EOF
provider no-ip.com {
	username = $DDNS_USER
	password = $DDNS_PASSWORD
	hostname = $HOSTNAME
}
EOF
			;;
		"dyndns")
			cat >> "$INADYN_CONF" <<EOF
provider dyndns.org {
	ssl	  = true
	username = $DDNS_USER
	password = $DDNS_PASSWORD
	hostname = $HOSTNAME
}
EOF
			;;
		"duckdns")
			cat >> "$INADYN_CONF" <<EOF
provider duckdns.org {
	username = $DDNS_PASSWORD
	password = noPasswordForDuckdns
	hostname = $HOSTNAME
}
EOF
			;;
		"cloudflare")
			cat >> "$INADYN_CONF" <<EOF
provider cloudflare.com {
	username = $DDNS_USER
	password = $DDNS_PASSWORD
	hostname = $HOSTNAME
	ttl = 1 # optional, value of 1 is 'automatic'.
	proxied = $DDNS_PROXIED
}
EOF
			;;
		*)
			echo "Unsupported DDNS service: $DDNS_SERVICE" >&2
			exit 1
			;;
	esac
else
	# ファイルアップロード処理
	if [ ! -f "$ABOSWEB_INADYN_CONF" ]; then
		echo "Error: Uploaded configuration file not found at $ABOSWEB_INADYN_CONF" >&2
		exit 1
	fi

	cp "$ABOSWEB_INADYN_CONF" "$INADYN_CONF"
fi

#inadynはuserで動く、ABOSWebも読むので644
chmod 644 "$INADYN_CONF"

#inadynがキャッシュを残すディレクトリをデフォルトの/var/cache/inadynから/var/app/volumes/inadynに変えておく
cat > "$INADYN_CONFD_CONF" <<EOF
cachedir="/var/app/volumes/inadyn"
EOF

#inadyn起動
rc-service inadyn restart
rc-update add inadyn

persist_file /etc/runlevels/default/inadyn \
	"$INADYN_CONF" \
	"$INADYN_CONFD_CONF"
