// SPDX-License-Identifier: MIT
#define VERSION "0.1.0_sig_forwarder"

#define _POSIX_C_SOURCE 1

#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <sys/prctl.h>
#include <sys/signalfd.h>
#include <sys/wait.h>
#include <unistd.h>

static const struct option long_options[] = {
	{ "help", no_argument, NULL, 'h' },
	{ "version", no_argument, NULL, 'V' },
	{ "debug", no_argument, NULL, 'd' },
	{ 0 }
};

int usage(char **argv, FILE *out, int rc)
{
	fprintf(out, "Usage: %s [-hP] [--] child [args...]\n",
		argv[0] ? argv[0] : "sig_forwarder");
	fprintf(out,
		"Spawns child command and forwards signals, reaping zombies\n\n");
	fprintf(out,
		"-P: run in pause mode (don't run anything, stop on TERM/INT)\n");
	fprintf(out, "-V, --version: version\n");
	fprintf(out, "-d, --debug: debug mode\n");
	fprintf(out, "-h, --help: this help\n");
	return rc;
}

int main(int argc, char *argv[])
{
	bool pause_mode = false;
	bool debug = false;

	while (true) {
		int c = getopt_long(argc, argv, "hPVd", long_options, NULL);
		if (c == -1)
			break;
		switch (c) {
		case 'h':
			return usage(argv, stdout, 0);
		case 'P':
			pause_mode = true;
			break;
		case 'V':
			printf("tini version " VERSION "\n");
			return 0;
		case 'd':
			debug = true;
			break;
		case '?':
			return usage(argv, stderr, 127);
		default:
			fprintf(stderr, "getopt return unexpected code 0%o\n",
				c);
			return 127;
		}
	}
	argv += optind;
	argc -= optind;
	// no argument for ourselves, but if nothing is passed bail out
	if (argc < 1 && !pause_mode) {
		return usage(argv, stderr, 127);
	}

	sigset_t sigmask, orig_sigmask;
	if (sigfillset(&sigmask)) {
		perror("sigfillset failed");
		return 127;
	}

	int sigfd = signalfd(-1, &sigmask, SFD_CLOEXEC);
	if (sigfd < 0) {
		perror("signalfd creation failed");
		return 127;
	}
	if (sigprocmask(SIG_BLOCK, &sigmask, &orig_sigmask) < 0) {
		perror("could not block signals");
		return 127;
	}

	if (debug) {
		// for tests, set ourself as main subreaper so commands such as
		// this get reaped directly: sig_forwarder sh -c '(true &); sleep 1'
		prctl(PR_SET_CHILD_SUBREAPER, 1, 0, 0, 0);
	}

	pid_t child = 0;
	if (!pause_mode) {
		child = fork();
		if (child < 0) {
			perror("fork failed");
			return 127;
		}
		if (child == 0) {
			if (sigprocmask(SIG_SETMASK, &orig_sigmask, NULL) < 0) {
				perror("Could not restore sigmask for child");
				return 127;
			}
			execvp(argv[0], argv);
			perror("exec failed");
			return 127;
		}
	}

	while (1) {
		ssize_t n;
		struct signalfd_siginfo siginfo;
		n = read(sigfd, &siginfo, sizeof(siginfo));
		if (n < 0 && errno == -EAGAIN)
			continue;
		if (n < 0) {
			perror("read failed");
			return 127;
		}
		if (n != sizeof(siginfo)) {
			errno = EINVAL;
			perror("read size wrong");
			return 127;
		}
		if (debug)
			printf("got signal %d\n", siginfo.ssi_signo);
		switch (siginfo.ssi_signo) {
		case SIGCHLD: {
			pid_t reaped;
			int wstatus;

			while ((reaped = waitpid(-1, &wstatus, WNOHANG)) > 0) {
				if (reaped != child)
					continue;
				if (WIFEXITED(wstatus))
					return WEXITSTATUS(wstatus);
				if (WIFSIGNALED(wstatus))
					return 128 + WTERMSIG(wstatus);
				return 127;
			}
			break;
		}
		case SIGTERM:
		case SIGINT:
		case SIGQUIT:
			if (pause_mode)
				return 0;
			// fallthrough
		default:
			// ignore any other signal in pause mode
			if (pause_mode)
				break;
			if (kill(child, siginfo.ssi_signo) < 0) {
				perror("could not forward %d to child: aborting");
				return 127;
			}
		}
	}

	return 127;
}
