// SPDX-License-Identifier: MIT
#define VERSION "0.1.1"

#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE

#include <stdlib.h>
#include <ctype.h>
#include <stdbool.h>
#include <getopt.h>
#include <string.h>
#include <strings.h>
#include <stdio.h>
#include <time.h>
#include <limits.h>

/* parse globals */
static bool debug;
static bool date_set;
static bool hour_set;
static bool second_set;
static bool cannot_set_hour;
static struct tm date_base;
static struct tm date_offset;
static char *token;
static char *datespec;
static int count = INT_MIN;
static int count_digits;
#define PARSE_SEPARATOR " \n\t"

#define countof(a) (sizeof(a) / sizeof(*(a)))

static const struct option long_options[] = {
	{ "help", no_argument, NULL, 'h' },
	{ "version", no_argument, NULL, 'V' },
	{ "debug", no_argument, NULL, 'd' },
	// undocumented option for tests
	{ "base-time", required_argument, NULL, 'B' },
	{ 0 }
};

static int usage(char **argv, FILE *out, int rc)
{
	fprintf(out, "Usage: %s [options] \"date spec\"\n",
		argv[0] ? argv[0] : "schedule_ts");
	fprintf(out, "Compute an unix timestamp from date spec\n\n");
	fprintf(out, "Option supported:\n");
	fprintf(out, "  -V, --version: print version and exit\n\n");
	fprintf(out, "  -d, --debug: enable debug messages\n\n");
	fprintf(out, "  -h, --help: this help\n\n");
	fprintf(out,
		"date specification aims to replicate a subset of GNU date format,\n");
	fprintf(out, "In particular, the following is supported:\n");
	fprintf(out, "  1 sec 2 min -> timestamp after 2mins and 1 second\n");
	fprintf(out,
		"  0 next month -> same day of next month at 0AM localtime\n");
	fprintf(out, "  3 next tuesday -> next tuesday at 3AM\n");
	fprintf(out, "  3 2 tuesday -> tuesday in two weeks, at 3AM\n");
	fprintf(out, "  tomorrow -> tomorrow at same time\n");
	fprintf(out, "  3 weeks -> same day of week in 3 weeks, same time\n");
	return rc;
}

static struct tm *normalize_date(struct tm *base, struct tm *offset)
{
	time_t time;
	static struct tm normalized;

	normalized = *base;
	normalized.tm_sec += offset->tm_sec;
	normalized.tm_min += offset->tm_min;
	normalized.tm_hour += offset->tm_hour;
	normalized.tm_mday += offset->tm_mday;
	normalized.tm_mon += offset->tm_mon;
	normalized.tm_year += offset->tm_year;
	time = mktime(&normalized);
	localtime_r(&time, &normalized);
	return &normalized;
}

#define trace(fmt, args...)                                           \
	do {                                                          \
		if (debug) {                                          \
			fprintf(stderr, "%s:%d: " fmt "\n", __func__, \
				__LINE__, ##args);                    \
		}                                                     \
	} while (0)
#define trace_time(fmt, args...)                                            \
	do {                                                                \
		if (debug) {                                                \
			char _buf[100];                                     \
			strftime(_buf, sizeof(_buf), "%Y-%m-%d %T",         \
				 normalize_date(&date_base, &date_offset)); \
			fprintf(stderr, "%s:%d: %s: " fmt "\n", __func__,   \
				__LINE__, _buf, ##args);                    \
		}                                                           \
	} while (0)

// weekdays as recognized by 'date'
static const struct weekday {
	const char *name;
	int wday;
} weekdays[] = {
	{ "sunday", 0 },   { "sun", 0 },  { "monday", 1 },   { "mon", 1 },
	{ "tuesday", 2 },  { "tues", 2 }, { "tue", 2 },	     { "wednesday", 3 },
	{ "wednes", 3 },   { "wed", 3 },  { "thursday", 4 }, { "thurs", 4 },
	{ "thur", 4 },	   { "thu", 4 },  { "friday", 5 },   { "fri", 5 },
	{ "saturday", 6 }, { "sat", 6 },
};

// months as recognized by 'date'
static const struct months {
	const char *name;
	int month;
} months[] = {
	{ "january", 0 },   { "jan", 0 },	{ "february", 1 },
	{ "feb", 1 },	    { "march", 2 },	{ "mar", 2 },
	{ "april", 3 },	    { "apr", 3 },	{ "may", 4 },
	{ "june", 5 },	    { "jun", 5 },	{ "july", 6 },
	{ "jul", 6 },	    { "august", 7 },	{ "aug", 7 },
	{ "september", 8 }, { "sept", 8 },	{ "sep", 8 },
	{ "october", 9 },   { "oct", 9 },	{ "november", 10 },
	{ "nov", 10 },	    { "december", 11 }, { "dec", 11 },
};

// 'numbers' as recognized by 'date'
static const struct numbers {
	const char *name;
	int num;
} numbers[] = {
	{ "last", -1 },
	{ "this", 0 },
	{ "next", 1 },
	{ "first", 1 },
	// second is also used in seconds, so not a number
	{ "third", 3 },
	{ "fourth", 4 },
	{ "fifth", 5 },
	{ "sixth", 6 },
	{ "seventh", 7 },
	{ "eight", 8 },
	{ "ninth", 9 },
	{ "tenth", 10 },
	{ "eleventh", 11 },
	{ "twelfth", 12 },
};

enum time_span {
	tm_sec,
	tm_min,
	tm_hour,
	tm_mday,
	tm_mon,
	tm_year,
};

static const struct timeshift {
	const char *name;
	int num;
	enum time_span span;
	bool can_count;
} timeshifts[] = {
	{ "yesterday", -1, tm_mday, false }, { "now", 0, tm_mday, false },
	{ "today", 0, tm_mday, false },	     { "tomorrow", 1, tm_mday, false },
	{ "second", 1, tm_sec, true },	     { "seconds", 1, tm_sec, true },
	{ "sec", 1, tm_sec, true },	     { "secs", 1, tm_sec, true },
	{ "minute", 1, tm_min, true },	     { "minutes", 1, tm_min, true },
	{ "min", 1, tm_min, true },	     { "mins", 1, tm_min, true },
	{ "hour", 1, tm_hour, true },	     { "hours", 1, tm_hour, true },
	{ "day", 1, tm_mday, true },	     { "days", 1, tm_mday, true },
	{ "week", 7, tm_mday, true },	     { "weeks", 7, tm_mday, true },
	{ "month", 1, tm_mon, true },	     { "months", 1, tm_mon, true },
	{ "year", 1, tm_year, true },	     { "years", 1, tm_year, true },
};

static const struct meridian {
	const char *name;
	int shift;
} meridians[] = {
	{ "AM", 0 },
	{ "A.M.", 0 },
	{ "PM", 12 },
	{ "P.M.", 12 },
};

static int fix_year(int year, int digits)
{
	if (digits != 2)
		return year - 1900;
	if (year < 70)
		return year + 100;
	if (year < 100)
		return year;
	return year - 1900;
}

static void set_hour(void)
{
	trace("Set_hour from %d", count);
	if (!hour_set) {
		date_base.tm_hour = 0;
		date_base.tm_min = 0;
	}
	if (!second_set)
		date_base.tm_sec = 0;
	if (count != INT_MIN) {
		if (cannot_set_hour) {
			fprintf(stderr,
				"Cannot set hour after +, missing unit?\n");
			exit(1);
		}
		if (count < 0) {
			fprintf(stderr, "negative hour?\n");
			exit(1);
		}
		if (date_set && hour_set) {
			date_base.tm_year = fix_year(count, count_digits);
		} else if (count_digits >= 5) {
			// date YYYYMMDD
			if (date_set) {
				fprintf(stderr, "Cannot set date twice\n");
				exit(1);
			}
			// BUG: 0xMMDD gets fixed, but xMMDD is taken as is...
			// cannot fix this easily at this point, we don't really
			// care
			date_base.tm_year = fix_year(count / 10000, count_digits - 4);
			date_base.tm_mon = (count % 10000) / 100 - 1;
			date_base.tm_mday = count % 100;
			date_set = true;
		} else if (count_digits >= 3) {
			// hour HHMM
			if (hour_set) {
				fprintf(stderr, "Cannot set hour twice\n");
				exit(1);
			}
			date_base.tm_hour += count / 100;
			date_base.tm_min += count % 100;
			hour_set = true;
		} else {
			// hour HH
			if (hour_set) {
				fprintf(stderr, "Cannot set hour twice\n");
				exit(1);
			}
			date_base.tm_hour += count;
			hour_set = true;
		}
		count = INT_MIN;
	}
	trace_time("hour set");
}

static void set_count(int c, int digits)
{
	if (count != INT_MIN)
		set_hour();
	count = c;
	count_digits = digits;
}

static int parse_meridian(bool commit)
{
	size_t i;
	int shift = -1;

	if (!token || !token[0])
		token = strtok(NULL, PARSE_SEPARATOR);
	if (!token) {
		if (commit)
			set_hour();
		return 1;
	}

	for (i = 0; i < countof(meridians); i++) {
		if (!strcasecmp(token, meridians[i].name)) {
			shift = meridians[i].shift;
			break;
		}
	}
	if (shift < 0) {
		if (commit)
			set_hour();
		return 0;
	}
	// ignore count_digits here: '003 am' is still '3am'...
	if (count < 100) {
		count *= 100;
		count_digits = 4;
	}
	if (count > 1259) {
		fprintf(stderr, "AM/PM can only be used with <= 12 hour\n");
		exit(1);
	}
	// 12PM = 12:00, 12AM = 00:00... offset back by 12.
	if (count >= 1200)
		count -= 1200;
	count += shift * 100;
	set_hour();
	return 1;
}

static int parse_count(void)
{
	// return 0 if skipping, 1 if token consumed
	size_t i;

	for (i = 0; i < countof(numbers); i++) {
		if (!strcasecmp(token, numbers[i].name)) {
			set_count(numbers[i].num, -1);
			return 1;
		}
	}

	long val;
	char *endp, *tmp;

	// TODO: if first char was a + it might be a timezone (+HHMM or +HH:MM)
	val = strtol(token, &endp, 10);
	if (endp == token)
		return 0;
	if (val > INT_MAX || val < INT_MIN)
		return 0;

	switch (*endp) {
	case 0:
		// was a full number, continue
		// TODO:... or peek at rest of string for DD MM YYYY, MM DD YYYY patterns
		// also DD MM / MM DD if and only if MM literal
		set_count(val, endp - token);
		token = NULL;
		return parse_meridian(false);
	case ':':
		// hour as HH:MM[:SS]
		// we shove HHMM in count to leave hour setting after am/pm is parsed,
		// but set seconds now if any
		if (val < 0 || val > 23) {
			fprintf(stderr, "Invalid token '%s'\n", token);
			exit(1);
		}
		set_count(val, endp - token);
		tmp = endp + 1;
		val = strtol(tmp, &endp, 10);
		if (tmp == endp || val < 0 || val > 59) {
			fprintf(stderr, "Invalid token '%s'\n", token);
			exit(1);
		}
		count = count * 100 + val;
		count_digits = 4;
		switch (*endp) {
		case ':':
			// got seconds
			tmp = endp + 1;
			val = strtol(endp + 1, &endp, 10);
			if (tmp == endp || val < 0 || val > 59) {
				fprintf(stderr, "Invalid token '%s'\n", token);
				exit(1);
			}
			date_base.tm_sec = val;
			second_set = true;
			break;
		}
		token = endp;
		return parse_meridian(true);
	case '/':
	case '-':
		// TODO: YYYY-MM-DD, MM numeric
		// TODO: DD-MM-YYYY, MM literal e.g. jan or january
		// TODO: MM/DD/YYYY or MM/DD
		// TODO: YYYY-MM-DDTHH:MM:SS.MS-TH:TM
		// TODO: YYYY-MM-DDTHH:MM:SS,NS-TH:TM
		// TODO: YYYY-MM-DD HH:MMZ
		// error for now
		fprintf(stderr, "- or / in '%s' not supported yet\n", token);
		exit(1);
	}
	if (isalpha(*endp)) {
		set_count(val, endp - token);
		token = endp;
		return parse_meridian(false);
	}

	// anything else was not valid, skip
	return 0;
}

static int parse_weekday(void)
{
	// return 1 if token consumed
	int wday = -1;
	size_t i;

	// TODO: A comma following a day of the week item is ignored.
	for (i = 0; i < countof(weekdays); i++) {
		if (!strcasecmp(token, weekdays[i].name)) {
			wday = weekdays[i].wday;
			break;
		}
	}
	if (wday == -1)
		return 0;

	int days = wday - date_base.tm_wday;
	// negative days offset get +7 (always in the future)
	if (days < 0)
		days += 7;

	if (count == INT_MIN) {
		// current weekday is no-op unless a count was given
		if (days == 0)
			return 1;
		// otherwise it defaults to 1
		count = 1;
	}
	// positive counts get -1 as 'next week' means we're not adding an extra 7 days,
	// except for current day of the week
	if (days > 0 && count > 0)
		count -= 1;

	trace("Adding %d days and %d weeks", days, count);
	// TODO: integer overflow on large counts
	date_offset.tm_mday += days + count * 7;
	count = INT_MIN;
	// this also resets hour in base date
	set_hour();

	return 1;
}

static int parse_month(void)
{
	// return 1 if token consumed
	int month = -1;
	size_t i;

	// TODO: A comma following a day of the week item is ignored.
	for (i = 0; i < countof(months); i++) {
		if (!strcasecmp(token, months[i].name)) {
			month = months[i].month;
			break;
		}
	}
	if (month == -1)
		return 0;

	trace("Found month %d", month);
	date_base.tm_mon = month;
	date_set = true;
	if (count != INT_MIN) {
		date_base.tm_mday = count;
		count = INT_MIN;

		// this also resets hour in base date
		set_hour();

		// peek if next token is a number for year
		token = strtok(NULL, PARSE_SEPARATOR);
		if (!token)
			return 1;

		char *endp;
		int val = strtol(token, &endp, 10);
		// TODO: check validity
		if (*endp != 0)
			return 0;
		// for some reason this date isn't "fixed" for 1970-2070 if < 100...
		date_base.tm_year = val - 1900;
	} else {
		// this also resets hour in base date
		set_hour();

		// peek if next token is a number for day
		token = strtok(NULL, PARSE_SEPARATOR);
		if (!token)
			return 1;

		char *endp;
		int val = strtol(token, &endp, 10);
		// TODO: check validity
		if (*endp != 0)
			return 0;
		date_base.tm_mday = val;
	}
	return 1;
}

static int parse_timeshift(void)
{
	// return 1 if token consumed
	const struct timeshift *shift = NULL;
	size_t i;

	for (i = 0; i < countof(timeshifts); i++) {
		if (!strcasecmp(token, timeshifts[i].name)) {
			shift = &timeshifts[i];
			break;
		}
	}
	if (!shift)
		return 0;

	if (!shift->can_count && count != INT_MIN)
		set_hour();

	if (count == INT_MIN)
		count = 1;

	// peek next word for 'ago'/'whence'
	token = strtok(NULL, PARSE_SEPARATOR);
	// already consumed token
	int rc = 0;

	if (token) {
		if (!strcasecmp(token, "ago")) {
			count = -count;
			rc = 1;
		} else if (!strcasecmp(token, "whence")) {
			rc = 1;
		}
	}

	switch (shift->span) {
	case tm_sec:
		date_offset.tm_sec += count * shift->num;
		break;
	case tm_min:
		date_offset.tm_min += count * shift->num;
		break;
	case tm_hour:
		date_offset.tm_hour += count * shift->num;
		break;
	case tm_mday:
		date_offset.tm_mday += count * shift->num;
		break;
	case tm_mon:
		date_offset.tm_mon += count * shift->num;
		break;
	case tm_year:
		date_offset.tm_year += count * shift->num;
		break;
	}
	count = INT_MIN;

	return rc;
}

/*
 * date parsing is separated in tokens e.g. '3am tomorrow' would first set hour to 3 am in
 * first token parsing then adjust date to tomorrow in the next one.
 */
static int parse_token(void)
{
	char *start_token;
	int rc;

	if (!token || !token[0]) {
		token = strtok(datespec, PARSE_SEPARATOR);
		datespec = NULL;
	}
	if (!token)
		return 0;

	trace_time("onto '%s'", token);
	start_token = token;

	if (token[0] == '+') {
		// only valid for one token
		cannot_set_hour = true;
		token++;
		rc = parse_token();
		cannot_set_hour = false;
		return rc;
	}

	rc = parse_count();
	if (rc) {
		// current token consumed, get next
		token = strtok(NULL, PARSE_SEPARATOR);
		if (!token) {
			set_hour();
			return 0;
		}
	}

	if (parse_weekday()) {
		token = NULL;
		return 1;
	}
	if (parse_month()) {
		token = NULL;
		return 1;
	}

	if (parse_timeshift()) {
		token = NULL;
		return 1;
	}

	if (token == start_token) {
		fprintf(stderr, "Didn't know how to parse '%s'\n", token);
		exit(1);
	}
	return 1;
}

static void normalize_datespec(void)
{
	// because we'll split the string in tokens later,
	// we do not want spaces around symbols before/after digits
	// e.g. 'foo - 3' should be 'foo -3' and '3 : 12' is '3:12'
	char *input = datespec;
	bool digit = false;
	bool symbol = false;
	bool modified = false;
	size_t len = strlen(datespec);

	while (input[0]) {
		if (isdigit(input[0])) {
			symbol = false;
			digit = true;
		} else if (strchr("-:/", input[0])) {
			symbol = true;
			digit = false;
		} else if (input[0] == ' ') {
			if (symbol && !digit) {
				// look ahead for digit
				char *next = input + 1;

				while (*next == ' ')
					next++;
				if (isdigit(next[0]))
					digit = true;
			}
			if (!symbol && digit) {
				// look ahead for symbol
				char *next = input + 1;

				while (*next == ' ')
					next++;
				if (strchr("-:/", next[0]))
					symbol = true;
			}
			if (symbol && digit) {
				// remove this space
				memmove(input, input + 1, len);
				modified = true;
				continue;
			}
		} else {
			symbol = false;
			digit = false;
		}
		input++;
		len--;
	}
	if (modified) {
		trace("Normalized to '%s'", datespec);
	}
}

static int parse_date(void)
{
	int rc = 1;

	trace_time("Parsing '%s'", datespec);
	normalize_datespec();
	while (rc > 0) {
		rc = parse_token();
	}

	trace_time("Final result");
	struct tm *date = normalize_date(&date_base, &date_offset);
	// cast to long long for 32 bit archs
	printf("%lld\n", (long long int)mktime(date));

	return 0;
}

int schedule_ts_main(int argc, char *argv[])
{
	time_t now = time(0);

	localtime_r(&now, &date_base);

	while (true) {
		int c = getopt_long(argc, argv, "hVdB:", long_options, NULL);

		if (c == -1)
			break;
		switch (c) {
		case 'h':
			return usage(argv, stdout, 0);
		case 'V':
			printf("schedule_ts version " VERSION "\n");
			return 0;
		case 'd':
			debug = true;
			break;
		case 'B': {
			char *endp = strptime(optarg, "%Y-%m-%d %H:%M:%S",
					      &date_base);

			if (!endp || endp[0] != '\0') {
				fprintf(stderr,
					"Invalid base time %s, must be %%Y-%%m-%%d %%H:%%M:%%S\n",
					optarg);
				return 127;
			}

			// musl bug: strptime does not set wday in date_base,
			// convert to time and back to fix it
			now = mktime(&date_base);
			localtime_r(&now, &date_base);
			break;
		}
		case '?':
			return usage(argv, stderr, 127);
		default:
			fprintf(stderr, "getopt return unexpected code 0%o\n",
				c);
			return 127;
		}
	}
	// require exactly one argument
	if (argc != optind + 1) {
		fprintf(stderr, "Require exactly one positional argument\n");
		return usage(argv, stderr, 127);
	}
	datespec = argv[optind];

	if (strcmp(datespec, "-")) {
		// real date to parse, otherwise get from stdin
		return parse_date();
	}

	struct tm base = date_base;
	char *line = NULL;
	int rc;
	size_t len;

	while (getline(&line, &len, stdin) >= 0) {
		datespec = line;
		date_base = base;
		date_set = hour_set = second_set = false;
		memset(&date_offset, 0, sizeof(date_offset));
		token = NULL;
		rc = parse_date();
		if (rc)
			return rc;
	}
	return 0;
}
